import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const answerKeysPath = path.join(__dirname, '../client/src/lib/answerKeys.json');
const answerKeys = JSON.parse(fs.readFileSync(answerKeysPath, 'utf-8'));

const STAKEHOLDER = "Startup (Founder)";

const allUpdates = {
  L0: {
    D1: {
      "1": "I show no evident passion; I treat this venture merely as a job.",
      "2": "I am interested in the problem, but I am easily distracted by other opportunities.",
      "3": "I have a clear personal connection to this problem and can articulate exactly why it matters to me.",
      "4": "I have deep, evident passion and am already investing my own time and money to solve this.",
      "5": "I would do this for free; this problem defines my identity."
    },
    D2: {
      "1": "I cannot articulate the problem; my thinking remains vague.",
      "2": "I can provide a basic explanation, but it lacks depth.",
      "3": "I give a clear explanation backed by a personal story.",
      "4": "I offer a compelling narrative with specific, tangible examples.",
      "5": "I tell a transformative story that demonstrates maturity and ecosystem vision."
    },
    D3: {
      "1": "I have no awareness of my values, and they are misaligned with this work.",
      "2": "I have vague alignment; inconsistency is visible in my actions.",
      "3": "I have a clear values statement that aligns with the problem I am solving.",
      "4": "I hold deeply held values, and my actions consistently reflect them.",
      "5": "I have a values-driven identity with complete alignment to the startup culture."
    },
    D4: {
      "1": "I have no firsthand experience; I rely on theory only.",
      "2": "I have indirect experience and only a surface-level understanding.",
      "3": "I have direct experience and have researched the problem well.",
      "4": "I have deep firsthand experience and understand the nuances.",
      "5": "I have lived expertise and possess the depth to mentor others through it."
    },
    D5: {
      "1": "I have a full-time job and no bandwidth for this.",
      "2": "I have limited time due to competing priorities.",
      "3": "I can dedicate 20–30 hours per week to this.",
      "4": "I can dedicate 40+ hours per week; I am fully committed.",
      "5": "I am all-in; I have restructured my entire life around this commitment."
    },
    D6: {
      "1": "I have no unique advantages; I am standard.",
      "2": "I have minor advantages, but I am not significantly different.",
      "3": "I have identified one clear advantage.",
      "4": "I have multiple advantages that give me a competitive edge.",
      "5": "I possess a rare combination of skills and insights that is my unique vantage point."
    },
    D7: {
      "1": "I cannot articulate a vision; I offer no inspiration.",
      "2": "I have a vague vision with limited power to inspire.",
      "3": "I have a clear vision that inspires some people.",
      "4": "I have a compelling vision that inspires many.",
      "5": "I am a visionary; I inspire movements that people are desperate to join."
    },
    D8: {
      "1": "I engage in short-term thinking; my motivation wanes quickly.",
      "2": "I have unclear longevity; I am at risk of burnout.",
      "3": "I am sustainable for 2–3 years; I have a good rhythm.",
      "4": "I am sustainable for 5–10 years; I can handle multiple cycles.",
      "5": "I have a lifelong commitment; I am crisis-resistant."
    },
    D9: {
      "1": "I am unwilling to sacrifice, or I am unaware of the need to do so.",
      "2": "I am aware of the need, but I am hesitant to commit.",
      "3": "I am willing to make moderate sacrifices.",
      "4": "I am already making sacrifices and am deeply committed.",
      "5": "I have made major life changes; I am in a state of all-in sacrifice."
    },
    EiR1: {
      "1": "I am defensive and lack self-awareness regarding my assumptions.",
      "2": "I can name one assumption, but I am dismissive of its impact.",
      "3": "I can name 2–3 assumptions and am thoughtful about them.",
      "4": "I have deep self-awareness and actively test my assumptions.",
      "5": "I am ruthlessly self-aware and constantly evolve my thinking."
    },
    EiR2: {
      "1": "I am blind to obvious risks.",
      "2": "I see one risk, but I miss others.",
      "3": "I identify key risks clearly.",
      "4": "I have deep risk awareness and mitigation strategies.",
      "5": "I see all risks and am fully prepared."
    },
    EiR3: {
      "1": "I am certain to experience catastrophic failure.",
      "2": "I am likely to fail if I lose my co-founder.",
      "3": "I could recover; I have a plan.",
      "4": "I have strong solo capability and would persevere.",
      "5": "I thrive independently and am driven regardless of others."
    },
    EiR4: {
      "1": "I would quit immediately.",
      "2": "I would panic and likely fail.",
      "3": "I could survive the competition.",
      "4": "I would adapt and compete effectively.",
      "5": "I love competition; I thrive on it."
    },
    EiR5: {
      "1": "I would abandon the venture immediately.",
      "2": "I would panic and be forced to pivot.",
      "3": "I could adapt my strategy.",
      "4": "I would adapt and continue with resilience.",
      "5": "I love pivoting; I thrive on change."
    },
    EiR6: {
      "1": "I have overlooked multiple critical barriers.",
      "2": "I have overlooked several barriers.",
      "3": "I have identified most barriers.",
      "4": "I know all major barriers.",
      "5": "I have lawyers thoroughly preparing me for all barriers."
    },
    EiR7: {
      "1": "No; it is impossible for me to survive.",
      "2": "No; I would likely fail quickly.",
      "3": "It is possible with extreme measures.",
      "4": "My situation is definitely survivable.",
      "5": "I could thrive on my own resources."
    },
    EiR8: {
      "1": "I am burning out already.",
      "2": "I have a strong burnout risk.",
      "3": "I am managing burnout fairly well.",
      "4": "I am preventing burnout well.",
      "5": "My energy is sustainable."
    },
    EiR9: {
      "1": "I am clearly chasing hype.",
      "2": "I am mostly hype-driven.",
      "3": "I am somewhat authentic, but there is some hype.",
      "4": "I have authentic motivation.",
      "5": "I have pure, authentic purpose."
    }
  },
  L1: {
    D1: {
      "1": "I have no validation; I rely on theory only.",
      "2": "I have talked to fewer than 5 people; it remains unclear.",
      "3": "I have talked to 20+ people; it is becoming clear.",
      "4": "I have had 100+ conversations; I have deep insight.",
      "5": "Demand exceeds supply; I have proven the problem exists."
    },
    D2: {
      "1": "I have done no market analysis.",
      "2": "I am guessing and largely optimistic.",
      "3": "I have done a basic TAM/SAM analysis.",
      "4": "I have done detailed market sizing.",
      "5": "I have a sophisticated TAM/SOM/SAM model."
    },
    D3: {
      "1": "I ignore feedback; I have a fixed mindset.",
      "2": "I listen, but I don't change.",
      "3": "Some iteration is happening.",
      "4": "I have a regular iteration cycle.",
      "5": "I rapidly iterate based on feedback."
    },
    D4: {
      "1": "I have generated no interest; zero.",
      "2": "I have fewer than 5 people interested.",
      "3": "I have 50+ people interested; there is some traction.",
      "4": "I have 100+ people interested; there is clear demand.",
      "5": "I have viral interest; demand is high."
    },
    D5: {
      "1": "I am burning fast with no validation.",
      "2": "I have minimal validation of my spend.",
      "3": "I am validating key assumptions.",
      "4": "I have a lean burn and high validation.",
      "5": "I have minimal burn and fully validated spend."
    },
    D6: {
      "1": "I have no special skills.",
      "2": "I have a generic background.",
      "3": "I have one clear strength.",
      "4": "I have multiple relevant strengths.",
      "5": "I have rare founder superpowers."
    },
    D7: {
      "1": "I have no transparency; I am hidden.",
      "2": "I do minimal sharing; I am opaque.",
      "3": "I do some public learning.",
      "4": "I engage in regular public iteration.",
      "5": "I maintain radical transparency on my journey."
    },
    D8: {
      "1": "I have no resilience; I quit.",
      "2": "I have weak resilience; I am fragile.",
      "3": "I have a normal resilience level.",
      "4": "I have high resilience; I bounce back.",
      "5": "I bounce back instantly."
    },
    D9: {
      "1": "I have no clear positioning.",
      "2": "I have vague positioning.",
      "3": "I have defined a clear positioning.",
      "4": "I have strong differentiation.",
      "5": "I have a unique, must-have position."
    },
    EiR1: {
      "1": "I assume I am always right.",
      "2": "I occasionally doubt, but I mostly assume I am right.",
      "3": "I test key assumptions.",
      "4": "I constantly challenge and test.",
      "5": "I use scientific validation."
    },
    EiR2: {
      "1": "I see all risks clearly.",
      "2": "I miss some risks, but I am aware of that possibility.",
      "3": "I miss several risks.",
      "4": "I am blind to many risks.",
      "5": "I am completely blind."
    },
    EiR3: {
      "1": "The team is solid.",
      "2": "We have some fragility, but it is manageable.",
      "3": "The team has weaknesses.",
      "4": "The team is at risk of breaking.",
      "5": "The team is crumbling."
    },
    EiR4: {
      "1": "There is no competition.",
      "2": "There is minor competition, but we have the advantage.",
      "3": "There are real competitors, but they are manageable.",
      "4": "There is strong competition and pressure.",
      "5": "There is an existential threat."
    },
    EiR5: {
      "1": "I am recession-proof; I am strong.",
      "2": "I am resilient; there is some impact.",
      "3": "I am vulnerable; I would struggle.",
      "4": "I am very vulnerable; I would fail.",
      "5": "I would perish in a downturn."
    },
    EiR6: {
      "1": "There is a clear path; there are no barriers.",
      "2": "There are minimal barriers; they are manageable.",
      "3": "There are some barriers; they are solvable.",
      "4": "There are major barriers; it is difficult.",
      "5": "There is a regulatory blockade."
    },
    EiR7: {
      "1": "I have low capital needs; I can bootstrap.",
      "2": "I can mostly bootstrap.",
      "3": "Some capital is needed.",
      "4": "Significant capital is needed.",
      "5": "I have a massive capital requirement."
    },
    EiR8: {
      "1": "We are fresh and energized.",
      "2": "We are mostly fresh; there is some fatigue.",
      "3": "There is some fatigue, but it is manageable.",
      "4": "There is significant fatigue.",
      "5": "We are experiencing severe burnout."
    },
    EiR9: {
      "1": "I have healthy validation.",
      "2": "I am mostly healthy, with some obsession.",
      "3": "I have some obsession.",
      "4": "I have significant obsession.",
      "5": "I have an unhealthy obsession."
    }
  },
  L2: {
    D1: {
      "1": "I have no MVP; I have a concept only.",
      "2": "I have a rough prototype; it is buggy.",
      "3": "I have a working MVP; I am learning.",
      "4": "I have a solid MVP; it is feature-complete.",
      "5": "I have a beautiful MVP; it is delightful."
    },
    D2: {
      "1": "I have no technical knowledge.",
      "2": "I have basic technical knowledge; there are gaps.",
      "3": "I have a good technical grasp.",
      "4": "I have deep technical expertise.",
      "5": "I have exceptional technical understanding."
    },
    D3: {
      "1": "I have no POC; it is speculation.",
      "2": "I have a weak POC; it is flawed.",
      "3": "I have a clear POC; it validates the concept.",
      "4": "I have a strong POC; it is impressive.",
      "5": "I have an exceptional POC."
    },
    D4: {
      "1": "I have no customer input.",
      "2": "I have minimal feedback; it is mostly ignored.",
      "3": "I engage in regular customer talks.",
      "4": "I have deep customer engagement.",
      "5": "I am customer-obsessed."
    },
    D5: {
      "1": "I engage in wasteful building.",
      "2": "I have some waste.",
      "3": "I have a lean approach.",
      "4": "I am very lean.",
      "5": "I am exceptionally lean."
    },
    D6: {
      "1": "I have no differentiation.",
      "2": "I have slight differentiation.",
      "3": "I have clear differentiation.",
      "4": "I have strong differentiation.",
      "5": "I have exceptional uniqueness."
    },
    D7: {
      "1": "I have no validation; I am guessing.",
      "2": "I have minimal validation.",
      "3": "I have some validation.",
      "4": "I have good validation.",
      "5": "I have deep validation."
    },
    D8: {
      "1": "We have critical gaps; we cannot build.",
      "2": "We have significant gaps.",
      "3": "We have some gaps; they are improvable.",
      "4": "We are mostly complete.",
      "5": "I have assembled a dream team."
    },
    D9: {
      "1": "I have no iteration; I am stuck.",
      "2": "I have slow iteration.",
      "3": "I have regular iteration.",
      "4": "I have rapid iteration.",
      "5": "I have obsessive iteration."
    },
    EiR1: {
      "1": "I assumed feasibility; it is untested.",
      "2": "I have done minimal testing.",
      "3": "I have done some testing.",
      "4": "I have done thorough testing.",
      "5": "My feasibility is scientifically proven."
    },
    EiR2: {
      "1": "I am blind to all debt.",
      "2": "I miss major debt.",
      "3": "I know most debt.",
      "4": "I know key debt.",
      "5": "I have identified all debt."
    },
    EiR3: {
      "1": "We have critical weakness.",
      "2": "We have significant weakness.",
      "3": "We have some gaps.",
      "4": "We have minor gaps.",
      "5": "We have no gaps."
    },
    EiR4: {
      "1": "It is already obsolete.",
      "2": "There is a risk of becoming obsolete.",
      "3": "There is moderate risk.",
      "4": "There is low risk.",
      "5": "It is future-proof."
    },
    EiR5: {
      "1": "It is not sustainable.",
      "2": "It is fragile.",
      "3": "It is sustainable.",
      "4": "It is highly sustainable.",
      "5": "It is perpetual."
    },
    EiR6: {
      "1": "I have no path; it is chaos.",
      "2": "I have a vague path.",
      "3": "I have a clear path.",
      "4": "I have a strong path.",
      "5": "I have perfect visibility."
    },
    EiR7: {
      "1": "I have no resources.",
      "2": "I have minimal resources.",
      "3": "I have good resources.",
      "4": "I have strong resources.",
      "5": "I have exceptional resources."
    },
    EiR8: {
      "1": "We have severe burnout.",
      "2": "We have significant fatigue.",
      "3": "We have some fatigue.",
      "4": "We are managing well.",
      "5": "We are fresh and sustainable."
    },
    EiR9: {
      "1": "It is pure ego/hype.",
      "2": "It is mostly ego; there is some substance.",
      "3": "I have mixed motives.",
      "4": "It is mostly mission-driven; there is some ego.",
      "5": "It is pure mission-driven."
    }
  },
  L3: {
    D1: {
      "1": "I have no revenue; zero users.",
      "2": "I have fewer than 100 users.",
      "3": "I have 100–1,000 users; there is traction.",
      "4": "I have 1,000+ users and growing.",
      "5": "I have rapid growth; clear Product-Market Fit (PMF)."
    },
    D2: {
      "1": "I have no customers.",
      "2": "I have fewer than 10.",
      "3": "I have 10–50 customers.",
      "4": "I have 50–200 customers.",
      "5": "I have 200+ customers."
    },
    D3: {
      "1": "I have no retention; pure churn.",
      "2": "I have poor retention.",
      "3": "I have moderate retention.",
      "4": "I have good retention.",
      "5": "I have excellent retention."
    },
    D4: {
      "1": "I have no virality.",
      "2": "I have minimal organic growth.",
      "3": "I have some organic growth.",
      "4": "I have a good viral loop.",
      "5": "I have viral growth."
    },
    D5: {
      "1": "I am losing on every sale.",
      "2": "I have poor unit economics.",
      "3": "I am breaking even.",
      "4": "I have good margins.",
      "5": "I have excellent margins."
    },
    D6: {
      "1": "I have no differentiation; \"me-too\".",
      "2": "I have slight differentiation.",
      "3": "I have clear differences.",
      "4": "I have strong differentiation.",
      "5": "I have unique positioning."
    },
    D7: {
      "1": "I have no momentum; I am stalled.",
      "2": "I have slow momentum.",
      "3": "I am building momentum.",
      "4": "I have strong momentum.",
      "5": "I have viral momentum."
    },
    D8: {
      "1": "I have no moat; I am vulnerable.",
      "2": "I have a weak moat.",
      "3": "I have a reasonable moat.",
      "4": "I have a good competitive moat.",
      "5": "I have a strong, defensible position."
    },
    D9: {
      "1": "I have no market position.",
      "2": "I have a marginal position.",
      "3": "I am an emerging leader.",
      "4": "I am a clear leader.",
      "5": "I am the market leader."
    },
    EiR1: {
      "1": "It is assumed and untested.",
      "2": "I have done minimal testing.",
      "3": "I have done some testing.",
      "4": "I have done thorough testing.",
      "5": "It is scientifically validated."
    },
    EiR2: {
      "1": "I am blind to competition.",
      "2": "I underestimate them.",
      "3": "I am aware of threats.",
      "4": "I have deep competitive awareness.",
      "5": "I have mapped all threats."
    },
    EiR3: {
      "1": "The team is falling apart.",
      "2": "There is significant tension.",
      "3": "There is some tension.",
      "4": "We have good cohesion.",
      "5": "We have excellent cohesion."
    },
    EiR4: {
      "1": "I am easily copied; I am doomed.",
      "2": "I have a high copy risk.",
      "3": "I have a moderate copy risk.",
      "4": "I have a low copy risk.",
      "5": "It is hard to copy me."
    },
    EiR5: {
      "1": "The market is oversaturated; it is dying.",
      "2": "The market is highly saturated.",
      "3": "The market is moderately saturated.",
      "4": "The market is growing.",
      "5": "It is an emerging market."
    },
    EiR6: {
      "1": "I am non-compliant; it is illegal.",
      "2": "I have major violations.",
      "3": "I have some issues.",
      "4": "I am mostly compliant.",
      "5": "I am fully compliant."
    },
    EiR7: {
      "1": "It is not sustainable.",
      "2": "Revenue is fragile.",
      "3": "It is somewhat sustainable.",
      "4": "It is sustainable.",
      "5": "It is highly sustainable."
    },
    EiR8: {
      "1": "We have severe burnout.",
      "2": "We have significant fatigue.",
      "3": "We have some fatigue.",
      "4": "We are managing well.",
      "5": "We are fresh and energized."
    },
    EiR9: {
      "1": "It is artificial traction; fake.",
      "2": "It is mostly hype/fake.",
      "3": "It is a mix of real and fake.",
      "4": "It is mostly organic.",
      "5": "It is pure organic."
    }
  },
  L4: {
    D1: {
      "1": "I am acquiring customers at a high cost with poor retention.",
      "2": "I have some CAC pressure; acquisition is inefficient.",
      "3": "I have balanced acquisition costs and retention.",
      "4": "I have efficient acquisition; low CAC.",
      "5": "I have exceptional unit economics; near-zero CAC."
    },
    D2: {
      "1": "I am confined to one geography or segment.",
      "2": "I am exploring adjacent segments; expansion is slow.",
      "3": "I have entered two–three new segments successfully.",
      "4": "I am scaling across multiple geographies and segments.",
      "5": "I have global reach; I am in all major markets."
    },
    D3: {
      "1": "My operations are chaotic; I cannot scale.",
      "2": "I have basic processes; they are fragile.",
      "3": "I have documented processes; they are repeatable.",
      "4": "I have optimized operations; they are scalable.",
      "5": "I have world-class operations; I am infinitely scalable."
    },
    D4: {
      "1": "I have a small team; I cannot delegate.",
      "2": "I have some managers; delegation is incomplete.",
      "3": "I have a management layer; growth is manageable.",
      "4": "I have strong managers; the team is growing.",
      "5": "I have built a world-class organization; it grows itself."
    },
    D5: {
      "1": "Sales is founder-dependent; it is not repeatable.",
      "2": "Sales is mostly founder-driven; there is some repeatability.",
      "3": "I have a repeatable sales process.",
      "4": "I have a scalable sales methodology.",
      "5": "I have a world-class, predictable sales machine."
    },
    D6: {
      "1": "I have no marketing; I rely on word-of-mouth.",
      "2": "I have basic marketing; results are unpredictable.",
      "3": "I have a marketing strategy; it is generating leads.",
      "4": "I have strong brand presence and marketing efficiency.",
      "5": "I have a recognized brand; I am a household name."
    },
    D7: {
      "1": "I am cash-negative; I am burning fast.",
      "2": "I have weak cash flow; I am dependent on funding.",
      "3": "I am approaching cash-positive; sustainability is in sight.",
      "4": "I am cash-positive and growing.",
      "5": "I am generating cash and reinvesting for growth."
    },
    D8: {
      "1": "I am losing to competition; my position is weakening.",
      "2": "I am under competitive pressure; my share is declining.",
      "3": "I am holding my competitive position.",
      "4": "I am gaining market share; I am becoming the leader.",
      "5": "I am the undisputed market leader; competition is insignificant."
    },
    D9: {
      "1": "I have no clear strategy; I am reacting to market.",
      "2": "I have a vague strategy; it is changing frequently.",
      "3": "I have a clear three–five-year strategy.",
      "4": "I have a well-defined strategy executed with discipline.",
      "5": "I have a visionary strategy that defines the market."
    },
    EiR1: {
      "1": "I assumed growth is linear; it has proven false.",
      "2": "I have discovered hidden scaling barriers.",
      "3": "I have identified key growth constraints.",
      "4": "I have tested my growth assumptions.",
      "5": "My growth model is scientifically validated."
    },
    EiR2: {
      "1": "Execution is breaking down; we are missing targets.",
      "2": "We have execution gaps; delivery is inconsistent.",
      "3": "Execution is reasonably consistent.",
      "4": "We execute our plans reliably.",
      "5": "We execute flawlessly at scale."
    },
    EiR3: {
      "1": "The organization is fragmented; silos are deepening.",
      "2": "There are significant organizational silos.",
      "3": "There is some organizational siloing.",
      "4": "We have good cross-functional alignment.",
      "5": "We have a unified, aligned organization."
    },
    EiR4: {
      "1": "I am dependent on one customer; I am vulnerable.",
      "2": "I have customer concentration risk.",
      "3": "I have moderate customer concentration.",
      "4": "I have diversified customers; risk is low.",
      "5": "I have broad customer distribution; I am resilient."
    },
    EiR5: {
      "1": "My technology cannot scale; architecture is breaking.",
      "2": "I have scaling bottlenecks that require redesign.",
      "3": "My technology scales with optimization.",
      "4": "My technology scales efficiently.",
      "5": "My technology is infinitely scalable."
    },
    EiR6: {
      "1": "Regulatory barriers are blocking growth.",
      "2": "I have regulatory friction; expansion is difficult.",
      "3": "I have identified regulatory pathways.",
      "4": "I am navigating regulations successfully.",
      "5": "I have compliant, scalable operations."
    },
    EiR7: {
      "1": "I cannot find talent; hiring is stalled.",
      "2": "I have significant hiring difficulties.",
      "3": "I am building my hiring pipeline.",
      "4": "I am hiring strong talent consistently.",
      "5": "I attract world-class talent effortlessly."
    },
    EiR8: {
      "1": "We have severe burnout and high attrition.",
      "2": "We have significant fatigue; people are leaving.",
      "3": "We have some attrition; it is manageable.",
      "4": "We have good retention; morale is healthy.",
      "5": "We have exceptional retention; we are thriving."
    },
    EiR9: {
      "1": "I am sacrificing everything for growth; profitability is impossible.",
      "2": "Growth is consuming resources; profits are negative.",
      "3": "I am balancing growth and profit.",
      "4": "I am achieving both growth and improving margins.",
      "5": "I am growing profitably; capital efficiency is exceptional."
    }
  },
  L5: {
    D1: {
      "1": "My unit economics are broken; I lose money on every transaction.",
      "2": "I have poor unit economics; margins are negative.",
      "3": "I have breakeven unit economics.",
      "4": "I have healthy unit economics; margins are improving.",
      "5": "I have exceptional unit economics; margins are expanding."
    },
    D2: {
      "1": "My LTV-to-CAC ratio is below 1:1; it is unsustainable.",
      "2": "My LTV-to-CAC ratio is 1.5:1; it is marginal.",
      "3": "My LTV-to-CAC ratio is 3:1; it is sustainable.",
      "4": "My LTV-to-CAC ratio is 5:1; it is healthy.",
      "5": "My LTV-to-CAC ratio is 8:1+; it is exceptional."
    },
    D3: {
      "1": "I have no operating leverage; costs scale with revenue.",
      "2": "I have limited operating leverage; COGS is high.",
      "3": "I have emerging operating leverage; margins improve with scale.",
      "4": "I have strong operating leverage; my SG&A ratio is declining.",
      "5": "I have exceptional operating leverage; my business gets cheaper to run as it grows."
    },
    D4: {
      "1": "My cash conversion cycle is long; I am financing customers.",
      "2": "My cash conversion cycle is slow; it is a drain.",
      "3": "My cash conversion cycle is manageable.",
      "4": "My cash conversion cycle is efficient; cash flows quickly.",
      "5": "My cash conversion cycle is negative; customers finance my growth."
    },
    D5: {
      "1": "My gross margins are below 20%; they are unsustainable.",
      "2": "My gross margins are 20–40%; they are improving.",
      "3": "My gross margins are 40–60%; they are healthy.",
      "4": "My gross margins are 60–80%; they are strong.",
      "5": "My gross margins are 80%+; they are exceptional."
    },
    D6: {
      "1": "I have no pricing power; I am a price-taker.",
      "2": "I have weak pricing power; price sensitivity is high.",
      "3": "I have moderate pricing power; I can adjust prices.",
      "4": "I have strong pricing power; customers are sticky.",
      "5": "I have exceptional pricing power; customers pay premium prices."
    },
    D7: {
      "1": "My costs are bloated; I am inefficient.",
      "2": "I have identified waste; optimization is beginning.",
      "3": "My costs are aligned with revenue.",
      "4": "I have optimized costs; I am lean.",
      "5": "I have engineered costs; every dollar is justified."
    },
    D8: {
      "1": "I require massive capital per dollar of value created.",
      "2": "I am capital-intensive; ROI is poor.",
      "3": "I have reasonable capital efficiency.",
      "4": "I am capital-efficient; I create value with little capital.",
      "5": "I am exceptionally capital-efficient; I create value from thin air."
    },
    D9: {
      "1": "I am deeply unprofitable; path to profit is unclear.",
      "2": "I have a path to profit, but it is distant.",
      "3": "I am approaching breakeven.",
      "4": "I am profitable and growing.",
      "5": "I am highly profitable with expanding margins."
    },
    EiR1: {
      "1": "Margins are compressing; prices are falling.",
      "2": "I have margin pressure; costs are rising.",
      "3": "Margins are stable.",
      "4": "I have margin expansion; leverage is improving.",
      "5": "My margins are widening; I have pricing power."
    },
    EiR2: {
      "1": "Larger scale is making me less efficient.",
      "2": "I am experiencing diseconomies at scale.",
      "3": "Economies and diseconomies are balanced.",
      "4": "I am experiencing strong economies of scale.",
      "5": "Scale is making me exponentially more efficient."
    },
    EiR3: {
      "1": "I am overleveraged; I am at risk of default.",
      "2": "I have high debt; leverage is concerning.",
      "3": "I have manageable debt levels.",
      "4": "I have conservative leverage; I have financial flexibility.",
      "5": "I have no debt; I am fully financially independent."
    },
    EiR4: {
      "1": "I am in a price war; margins are collapsing.",
      "2": "I am under price pressure from competitors.",
      "3": "I can maintain prices against competition.",
      "4": "I have pricing power; I raise prices when I choose.",
      "5": "My pricing is uncontested; I set market prices."
    },
    EiR5: {
      "1": "My churn is high; cohort economics are negative.",
      "2": "My churn is concerning; LTV is at risk.",
      "3": "My churn is healthy; retention is improving.",
      "4": "My churn is low; LTV is expanding.",
      "5": "My churn is near-zero; customers are for life."
    },
    EiR6: {
      "1": "I have unexpected tax and regulatory costs eroding profit.",
      "2": "I have significant tax and compliance burden.",
      "3": "I have planned for taxes and regulations.",
      "4": "I have optimized my tax and regulatory position.",
      "5": "I am fully tax-optimized; my structure is ideal."
    },
    EiR7: {
      "1": "Working capital swings are destroying cash flow.",
      "2": "I have working capital strain; cash is tight.",
      "3": "Working capital is manageable.",
      "4": "I have optimized working capital; cash is healthy.",
      "5": "Working capital is negative; I am self-funding."
    },
    EiR8: {
      "1": "The team is burned out; execution is slipping.",
      "2": "We have fatigue; quality is declining.",
      "3": "We are managing fatigue reasonably.",
      "4": "We are executing well; morale is healthy.",
      "5": "We are executing flawlessly; the team is thriving."
    },
    EiR9: {
      "1": "I am sacrificing customers and culture for profit.",
      "2": "Profit optimization is harming customer satisfaction.",
      "3": "I am balancing profit with stakeholder needs.",
      "4": "I am achieving profit with strong customer loyalty.",
      "5": "I am maximizing profit while strengthening culture and customer loyalty."
    }
  },
  L6: {
    D1: {
      "1": "I have no culture; it is chaotic and reactive.",
      "2": "I have a fragile culture; it is personality-dependent.",
      "3": "I have a defined culture; it is becoming embedded.",
      "4": "I have a strong, distinctive culture.",
      "5": "I have built a world-class culture; it is self-sustaining."
    },
    D2: {
      "1": "I have no management layers; I do everything.",
      "2": "I have one layer of management; it is thin.",
      "3": "I have two–three layers of management; depth is emerging.",
      "4": "I have deep management layers; succession is planned.",
      "5": "I have world-class management depth; I am expendable."
    },
    D3: {
      "1": "I have no innovation; I am maintaining the status quo.",
      "2": "Innovation is ad hoc; I am not systematic.",
      "3": "I have a basic innovation process; new ideas are explored.",
      "4": "I have a strong innovation pipeline; new products are launched regularly.",
      "5": "I have a world-class innovation engine; I define the future."
    },
    D4: {
      "1": "I have no board; I am an autocracy.",
      "2": "I have advisors, but no real governance.",
      "3": "I have a functional board; governance is emerging.",
      "4": "I have a strong board; governance is professional.",
      "5": "I have an exceptional board; it steers strategy."
    },
    D5: {
      "1": "I have no customer feedback mechanism; I am disconnected.",
      "2": "I have informal customer input; it is inconsistent.",
      "3": "I have a customer advisory board; input is regular.",
      "4": "I have a strong customer council; it influences strategy.",
      "5": "I have a world-class customer advisory network; they co-create with me."
    },
    D6: {
      "1": "I make decisions based on gut feeling; data is ignored.",
      "2": "I have data, but I often override it.",
      "3": "I am increasingly data-informed; my decisions are evidence-based.",
      "4": "I am data-driven; decisions are rigorously tested.",
      "5": "I am obsessively data-driven; every decision is quantified."
    },
    D7: {
      "1": "I am isolated from stakeholders; I do not engage.",
      "2": "I have minimal stakeholder engagement.",
      "3": "I engage regularly with key stakeholders.",
      "4": "I have strong stakeholder relationships; influence is clear.",
      "5": "I am a thought leader; stakeholders seek my counsel."
    },
    D8: {
      "1": "I resist change; I am rigid.",
      "2": "I adapt slowly; change is painful.",
      "3": "I am adaptable; I respond to change.",
      "4": "I embrace change; I am agile.",
      "5": "I am antifragile; I grow stronger from change."
    },
    D9: {
      "1": "I have no vision beyond next quarter.",
      "2": "I have a short-term focus; vision is unclear.",
      "3": "I have a multi-year vision; it is communicated.",
      "4": "I have a compelling long-term vision; it inspires.",
      "5": "I have a transformative vision; I am defining the future."
    },
    EiR1: {
      "1": "I have no succession plan; I am irreplaceable.",
      "2": "Succession planning is minimal; continuity is at risk.",
      "3": "I have identified potential successors.",
      "4": "I am developing successors actively.",
      "5": "My succession is planned; the company thrives without me."
    },
    EiR2: {
      "1": "My culture is toxic; people are leaving.",
      "2": "My culture has problems; morale is declining.",
      "3": "My culture is stable; issues are being addressed.",
      "4": "My culture is healthy; people thrive.",
      "5": "My culture is exceptional; it is my competitive advantage."
    },
    EiR3: {
      "1": "Innovation has stalled; we are falling behind.",
      "2": "Innovation is inconsistent; competitors are catching up.",
      "3": "Innovation is happening; we are keeping pace.",
      "4": "Innovation is strong; we are pulling ahead.",
      "5": "Innovation is exceptional; we define the industry."
    },
    EiR4: {
      "1": "My board is dysfunctional; it creates friction.",
      "2": "Board relations are strained; alignment is poor.",
      "3": "Board relations are professional; it functions.",
      "4": "Board relations are strong; it adds value.",
      "5": "My board is exceptional; it is my greatest asset."
    },
    EiR5: {
      "1": "I am disconnected from customers; I am losing touch.",
      "2": "Customer connection is weakening; feedback is rare.",
      "3": "Customer connection is adequate; I receive feedback.",
      "4": "Customer connection is strong; I understand their needs.",
      "5": "Customer connection is exceptional; I co-create with them."
    },
    EiR6: {
      "1": "My decisions are arbitrary; there is no process.",
      "2": "Decision-making is inconsistent; outcomes vary.",
      "3": "Decision-making is reasonable; processes exist.",
      "4": "Decision-making is strong; outcomes are predictable.",
      "5": "Decision-making is exceptional; I make consistently great decisions."
    },
    EiR7: {
      "1": "I am an acquisition target; my independence is threatened.",
      "2": "I am vulnerable to acquisition; large competitors could buy me.",
      "3": "I could be acquired if a buyer emerges.",
      "4": "I am independent; acquisition is unlikely unless strategic.",
      "5": "I am a predator; I am acquiring others, not being acquired."
    },
    EiR8: {
      "1": "I am burned out; I cannot lead effectively.",
      "2": "I am fatigued; my judgment is compromised.",
      "3": "I am managing fatigue; my energy is reasonably sustained.",
      "4": "I am energized; I lead with clarity.",
      "5": "I am thriving; my energy is boundless."
    },
    EiR9: {
      "1": "I am obsessed with my vision; reality is ignored.",
      "2": "My vision is sometimes divorced from reality.",
      "3": "I balance vision with pragmatism.",
      "4": "My vision is grounded in reality; execution is clear.",
      "5": "My vision is both inspiring and grounded; I move mountains."
    }
  },
  L7: {
    D1: {
      "1": "I have a niche position; I am not dominant.",
      "2": "I am a strong player; dominance is distant.",
      "3": "I am gaining market share; I am a clear leader.",
      "4": "I am the market leader; my position is uncontested.",
      "5": "I define the market; I am the undisputed category king."
    },
    D2: {
      "1": "I have minimal brand recognition; I am unknown.",
      "2": "My brand is recognized within my segment.",
      "3": "My brand is recognized broadly; it is a household name within my industry.",
      "4": "My brand is iconic; it is synonymous with the category.",
      "5": "My brand transcends business; it is a cultural phenomenon."
    },
    D3: {
      "1": "I have no impact on my ecosystem; I am extractive.",
      "2": "I am beginning to shape my ecosystem.",
      "3": "I am a key player; my ecosystem is healthier because of me.",
      "4": "I am shaping my ecosystem; I define standards.",
      "5": "I am the center of my ecosystem; industry follows my lead."
    },
    D4: {
      "1": "I struggle to attract talent; candidates avoid me.",
      "2": "I attract some talent, but competition is fierce.",
      "3": "I attract strong talent; I am an attractive employer.",
      "4": "I am a talent magnet; top talent seeks me out.",
      "5": "I am the most desired employer in my field; the best talent competes to join."
    },
    D5: {
      "1": "I struggle to raise capital; investors are skeptical.",
      "2": "I raise capital, but I must work hard for it.",
      "3": "Capital is available; investors are interested.",
      "4": "Capital seeks me out; I am a founder's dream.",
      "5": "I have unlimited capital access; investors compete for the opportunity."
    },
    D6: {
      "1": "I have no strategic partnerships; I am isolated.",
      "2": "I have some partnerships; they are transactional.",
      "3": "I have meaningful partnerships; they create value.",
      "4": "I have deep strategic partnerships; they amplify my impact.",
      "5": "I am at the center of a network of mutually beneficial partnerships."
    },
    D7: {
      "1": "I have no thought leadership voice; I am unknown.",
      "2": "I occasionally share ideas; influence is minimal.",
      "3": "I am recognized as a thought leader in my space.",
      "4": "I am an influential voice; industry listens to me.",
      "5": "I define industry thinking; I am the oracle of my field."
    },
    D8: {
      "1": "I am fragile; market shifts could destroy me.",
      "2": "I am vulnerable; competition could overtake me.",
      "3": "I am resilient; I can weather market cycles.",
      "4": "I am antifragile; adversity strengthens me.",
      "5": "I am immortal; I am built to last centuries."
    },
    D9: {
      "1": "I have no societal impact; I am focused on profit.",
      "2": "I have minimal societal impact; it is peripheral.",
      "3": "I am making a meaningful difference in society.",
      "4": "I am creating systemic positive change.",
      "5": "I have fundamentally transformed society; my legacy is immortal."
    },
    EiR1: {
      "1": "I am vulnerable to disruption; new tech could destroy me.",
      "2": "Disruption is a risk; I am not prepared.",
      "3": "I am monitoring disruption; I have plans.",
      "4": "I am ahead of disruption; I am the disruptor.",
      "5": "I am invulnerable to disruption; I define the future."
    },
    EiR2: {
      "1": "I am complacent; I am resting on laurels.",
      "2": "I am showing signs of stagnation.",
      "3": "I am pushing for innovation; I am avoiding complacency.",
      "4": "I maintain hunger; I keep innovating.",
      "5": "I am insatiably ambitious; I will never rest."
    },
    EiR3: {
      "1": "I am at risk of antitrust action; regulators are targeting me.",
      "2": "I have antitrust risk; regulatory scrutiny is high.",
      "3": "I am compliant; antitrust risk is manageable.",
      "4": "I am working with regulators; my position is stable.",
      "5": "I have a stable regulatory relationship; I define the rules."
    },
    EiR4: {
      "1": "I am losing talent; retention is poor.",
      "2": "Talent retention is challenging; attrition is rising.",
      "3": "Talent retention is reasonable; some churn is natural.",
      "4": "I retain top talent; loyalty is high.",
      "5": "My talent never leaves; loyalty is absolute."
    },
    EiR5: {
      "1": "My culture is diluted; I no longer recognize the company.",
      "2": "My culture is changing; values are shifting.",
      "3": "My culture is evolving; core values remain.",
      "4": "My culture is strengthening at scale; values are embedded.",
      "5": "My culture transcends scale; it is indestructible."
    },
    EiR6: {
      "1": "My market is saturated; growth is impossible.",
      "2": "My market is maturing; growth is slowing.",
      "3": "I can expand into adjacent markets.",
      "4": "I have multiple growth vectors; expansion is strong.",
      "5": "My addressable market is infinite; growth is limitless."
    },
    EiR7: {
      "1": "I have conflict between shareholders and stakeholders.",
      "2": "There is tension between profit and purpose.",
      "3": "I balance shareholder and stakeholder interests.",
      "4": "I align shareholder and stakeholder interests.",
      "5": "My purpose creates alignment; all stakeholders benefit."
    },
    EiR8: {
      "1": "My board and I are in conflict; control is contested.",
      "2": "There is friction with my board; alignment is unclear.",
      "3": "I have a functional relationship with my board.",
      "4": "I have a strong partnership with my board.",
      "5": "My board is my greatest asset; we are perfectly aligned."
    },
    EiR9: {
      "1": "My legacy is being distorted; I have lost control of the narrative.",
      "2": "My legacy is contested; different interpretations exist.",
      "3": "My legacy is reasonably understood.",
      "4": "My legacy is clearly articulated; it is well understood.",
      "5": "My legacy transcends debate; it is universally recognized and revered."
    }
  },
  L8: {
    D1: {
      "1": "I am damaging the environment; I am extractive.",
      "2": "I am not prioritizing environmental responsibility.",
      "3": "I am making environmental improvements; impact is positive.",
      "4": "I am carbon-neutral; environmental footprint is minimized.",
      "5": "I am carbon-negative; I am healing the environment."
    },
    D2: {
      "1": "I am extracting from society; I am widening inequity.",
      "2": "I have minimal social impact; equity is not a priority.",
      "3": "I am making positive social contributions.",
      "4": "I am systematically advancing equity and inclusion.",
      "5": "I am transforming society; I am healing historical inequities."
    },
    D3: {
      "1": "I have no transparency; I am opaque.",
      "2": "I provide minimal transparency; stakeholders distrust me.",
      "3": "I am transparent; stakeholders understand my operations.",
      "4": "I am radically transparent; governance is exemplary.",
      "5": "I am the gold standard of transparency; I define governance best practices."
    },
    D4: {
      "1": "I am exploiting my employees; wellbeing is ignored.",
      "2": "I have minimal commitment to employee wellbeing.",
      "3": "I am investing in employee health and development.",
      "4": "I have world-class benefits; employee wellbeing is exceptional.",
      "5": "I am creating a thriving human community; my employees transcend job satisfaction."
    },
    D5: {
      "1": "My supply chain is unethical; I turn a blind eye.",
      "2": "I have minimal oversight of supply chain ethics.",
      "3": "I am monitoring my supply chain; ethical practices are improving.",
      "4": "My supply chain is ethical; I audit and enforce standards.",
      "5": "My supply chain is exemplary; partners share my values completely."
    },
    D6: {
      "1": "I hoard knowledge; I contribute nothing.",
      "2": "I reluctantly share knowledge; it is strategic only.",
      "3": "I share knowledge selectively; ecosystem benefits are emerging.",
      "4": "I am open-sourcing innovations; I am advancing the industry.",
      "5": "I am a knowledge commons; I am lifting the entire ecosystem."
    },
    D7: {
      "1": "I have no succession plan; the company dies with me.",
      "2": "Succession planning is ad hoc; continuity is uncertain.",
      "3": "I have identified a successor; transition is planned.",
      "4": "I have groomed and transitioned leadership; it is thriving.",
      "5": "My successor has exceeded me; my transition is complete and transcendent."
    },
    D8: {
      "1": "My company will not survive me; it is fragile.",
      "2": "My company may survive; permanence is uncertain.",
      "3": "My company is built to endure; it has 10+ year runway.",
      "4": "My company is built for the long-term; it will outlive me.",
      "5": "My company is immortal; it is a permanent institution."
    },
    D9: {
      "1": "I exist for profit; purpose is absent.",
      "2": "Purpose is secondary to profit.",
      "3": "I balance profit with purpose.",
      "4": "Purpose drives profit; they are aligned.",
      "5": "I am pure purpose; profit is a consequence."
    },
    EiR1: {
      "1": "My mission has been completely abandoned.",
      "2": "My mission is being diluted; I am losing focus.",
      "3": "My mission remains clear; drift is minimal.",
      "4": "My mission is unwavering; I am fully aligned.",
      "5": "My mission is absolute; I would die before abandoning it."
    },
    EiR2: {
      "1": "I have ethical decay; corruption is visible.",
      "2": "I have ethical vulnerabilities; shortcuts are being taken.",
      "3": "I am maintaining ethical standards; vigilance is active.",
      "4": "I have strong ethical culture; corruption is unthinkable.",
      "5": "I am ethically incorruptible; integrity is uncompromising."
    },
    EiR3: {
      "1": "I am captured by regulators; I have lost independence.",
      "2": "Regulatory pressure is constraining my autonomy.",
      "3": "I work with regulators; independence is maintained.",
      "4": "I influence regulations; I shape policy.",
      "5": "I define regulations; I am the gold standard."
    },
    EiR4: {
      "1": "My business creates existential societal risk.",
      "2": "My business has significant societal downsides.",
      "3": "My business is neutral to society.",
      "4": "My business creates net positive societal value.",
      "5": "My business is essential to human flourishing."
    },
    EiR5: {
      "1": "My mortality is a threat to the company; death would destroy it.",
      "2": "My successor is untested; continuity is uncertain.",
      "3": "My successor is ready; transition plans exist.",
      "4": "My successor is proven; the company will thrive without me.",
      "5": "My legacy is immortal; I am transcendent."
    },
    EiR6: {
      "1": "I am overreaching; hubris is destroying me.",
      "2": "I am showing signs of hubris; ego is inflating.",
      "3": "I remain humble; ego is in check.",
      "4": "I am grounded; I know my limits.",
      "5": "I am humble and limitless; I have transcended ego entirely."
    },
    EiR7: {
      "1": "My business depends on systems that are collapsing.",
      "2": "Climate risk is significant; I have no mitigation.",
      "3": "I am planning for climate change; resilience is building.",
      "4": "I am climate-resilient; I am adapting.",
      "5": "I am climate-positive; I am healing the planet."
    },
    EiR8: {
      "1": "I am vulnerable to geopolitical disruption; I am unprepared.",
      "2": "Geopolitical risk is a concern; mitigation is minimal.",
      "3": "I have identified geopolitical risks; I have contingencies.",
      "4": "I am geopolitically resilient; I can navigate instability.",
      "5": "I am geopolitically antifragile; conflict strengthens me."
    },
    EiR9: {
      "1": "I have lost meaning; I am empty.",
      "2": "I sometimes question my purpose; fulfillment is uncertain.",
      "3": "I have found meaning; fulfillment is present.",
      "4": "I am deeply fulfilled; my life has profound meaning.",
      "5": "I have transcended meaning; I am one with my purpose."
    }
  }
};

let updateCount = 0;
let missingCount = 0;

for (const [levelId, dimensions] of Object.entries(allUpdates)) {
  if (!answerKeys[levelId]) {
    console.log(`Level ${levelId} not found`);
    continue;
  }
  
  for (const [dimId, grades] of Object.entries(dimensions)) {
    if (!answerKeys[levelId].questions?.[dimId]) {
      console.log(`${levelId} ${dimId} not found in answerKeys`);
      missingCount++;
      continue;
    }
    
    const stakeholderData = answerKeys[levelId].questions[dimId].stakeholders?.[STAKEHOLDER];
    if (!stakeholderData) {
      console.log(`${levelId} ${dimId} - Stakeholder ${STAKEHOLDER} not found`);
      missingCount++;
      continue;
    }
    
    for (const [grade, answer] of Object.entries(grades)) {
      stakeholderData.grades[grade] = answer;
    }
    updateCount++;
    
    console.log(`Updated ${levelId} ${dimId} (5 grades)`);
  }
}

fs.writeFileSync(answerKeysPath, JSON.stringify(answerKeys, null, 2), 'utf-8');

console.log(`\n✅ Updated ${updateCount} Startup Founder entries with all 5 grades.`);
console.log(`⚠️ Missing entries: ${missingCount}`);
console.log(`Total individual grade updates: ${updateCount * 5}`);
console.log(`File saved to: ${answerKeysPath}`);
