import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const answerKeysPath = path.join(__dirname, '../client/src/lib/answerKeys.json');
const answerKeys = JSON.parse(fs.readFileSync(answerKeysPath, 'utf-8'));

const STAKEHOLDER = "Investor";

const allUpdates = {
  L0: {
    D1: {
      "1": "I cannot spot talent; I use the wrong criteria.",
      "2": "I find pitched founders but miss raw talent.",
      "3": "I identify 1–2 diamonds; I have a track record.",
      "4": "I consistently find pre-traction founders.",
      "5": "I am legendary for finding founders early."
    },
    D2: {
      "1": "I dismiss passion; I focus only on confidence.",
      "2": "I value passion but am easily fooled.",
      "3": "I have a clear passion detector; I assess staying power.",
      "4": "I have a sophisticated assessment; I track well.",
      "5": "I have a legendary passion gauge; I am never wrong."
    },
    D3: {
      "1": "My criteria are biased; I miss resilience.",
      "2": "My criteria exist but miss important signals.",
      "3": "I have clear resilience criteria; I have a proven eye.",
      "4": "I have sophisticated criteria; I have predictive accuracy.",
      "5": "My criteria are legendary; I never back weak founders."
    },
    D4: {
      "1": "I have no early exits; my portfolio failed.",
      "2": "I have few exits; mostly failures.",
      "3": "I have several exits; I am learning the pattern.",
      "4": "I have multiple exits; I have repeatable success.",
      "5": "I am a legendary investor; my early bets yield 10x+."
    },
    D5: {
      "1": "I have no early-stage capital; I invest late only.",
      "2": "I have a small allocation; it is insufficient.",
      "3": "I have meaningful early-stage capital ready.",
      "4": "I have a dedicated early-stage fund; I am patient.",
      "5": "I am a legendary deployer; I have deep pockets."
    },
    D6: {
      "1": "My network is generic; I see what everyone sees.",
      "2": "I have some network; I have limited advantage.",
      "3": "I have a strong founder network for sourcing.",
      "4": "I have an exceptional network; I see pre-traction.",
      "5": "I have a legendary network; I find hidden gems."
    },
    D7: {
      "1": "I provide capital only; I have no value-add.",
      "2": "I do some advising; it has limited impact.",
      "3": "I offer clear value-add and helpful mentoring.",
      "4": "I offer exceptional value-add; I open doors.",
      "5": "My value-add is legendary; I transform trajectories."
    },
    D8: {
      "1": "My processes are bureaucratic and slow.",
      "2": "I have some slowness; I am occasionally responsive.",
      "3": "My processes are lean and responsive.",
      "4": "I am exceptionally lean; I move startup-fast.",
      "5": "I move with lightning speed; founders praise me."
    },
    D9: {
      "1": "I cannot discuss vulnerability; I am distant.",
      "2": "I acknowledge it but am uncomfortable discussing.",
      "3": "I can mentor with empathetic guidance.",
      "4": "I offer deep vulnerability mentoring.",
      "5": "I am legendary for walking through the darkness."
    },
    EiR1: {
      "1": "I confuse passion with traction; I bet wrong.",
      "2": "I bet on passion alone; I do no diligence.",
      "3": "I validate ideas and assess passion simultaneously.",
      "4": "I use rigorous validation; I rely less on passion.",
      "5": "I am legendary for idea validation; I test the premise."
    },
    EiR2: {
      "1": "I ignore competitive threats; I am blind.",
      "2": "I acknowledge threats but underestimate them.",
      "3": "I conduct clear competitive analysis.",
      "4": "I have sophisticated competition mapping.",
      "5": "I am a legend for predicting disruption."
    },
    EiR3: {
      "1": "I avoid conflict; I am a weak leader.",
      "2": "I handle conflict poorly; I navigate poorly.",
      "3": "I navigate conflict decently; I reduce friction.",
      "4": "I am an expert conflict navigator.",
      "5": "I am a legend for tough negotiations."
    },
    EiR4: {
      "1": "My controls stifle innovation.",
      "2": "I have some control constraints.",
      "3": "I allow reasonable innovation freedom.",
      "4": "I enable full innovation freedom.",
      "5": "I am a legend for enabling innovation."
    },
    EiR5: {
      "1": "I penalize pivots; I am rigid.",
      "2": "I tolerate pivots reluctantly.",
      "3": "I allow good pivots; I adapt.",
      "4": "I enable smart pivoting strategies.",
      "5": "I am a legend for pivot strategy."
    },
    EiR6: {
      "1": "My network is totally closed.",
      "2": "My network is mostly closed.",
      "3": "I am expanding my network somewhat.",
      "4": "I have an open, diverse network.",
      "5": "I am a legend for diverse sourcing."
    },
    EiR7: {
      "1": "I disappear in uncertainty.",
      "2": "I provide limited support.",
      "3": "I provide some support.",
      "4": "I provide strong support through doubt.",
      "5": "I am a legend for crisis support."
    },
    EiR8: {
      "1": "I am fatigued and disconnected.",
      "2": "I show some fatigue signs.",
      "3": "I am managing my fatigue okay.",
      "4": "I am preventing fatigue well.",
      "5": "My compassion is sustainable."
    },
    EiR9: {
      "1": "I am clearly chasing hype.",
      "2": "I am mostly hype-driven.",
      "3": "I am somewhat authentic; mixed motives.",
      "4": "I have authentic motivation.",
      "5": "I invest with pure, authentic purpose."
    }
  },
  L1: {
    D1: {
      "1": "I cannot identify traction; I rely on no metrics.",
      "2": "I look at vanity metrics only.",
      "3": "I understand real traction signals.",
      "4": "I am an expert at spotting momentum.",
      "5": "I am a legendary traction spotter."
    },
    D2: {
      "1": "I cannot distinguish markets from hype.",
      "2": "I confuse hype with markets.",
      "3": "I can identify real markets.",
      "4": "I am an expert at market validation.",
      "5": "I am a legend at spotting real markets."
    },
    D3: {
      "1": "I am rigid; I do not pivot.",
      "2": "I am slow to pivot.",
      "3": "I am flexible; I make some pivots.",
      "4": "I am quick to pivot based on data.",
      "5": "I am legendary for smart pivots."
    },
    D4: {
      "1": "I back no early signals.",
      "2": "I back a few with early signals.",
      "3": "I back several founders with traction.",
      "4": "I back multiple founders with momentum.",
      "5": "I am a legendary early traction spotter."
    },
    D5: {
      "1": "I expect quick ROI; I am impatient.",
      "2": "I have some patience; it is limited.",
      "3": "I am patient with early validation.",
      "4": "I am very patient; I provide a long runway.",
      "5": "I provide legendary, patient capital."
    },
    D6: {
      "1": "My network is generic.",
      "2": "I have limited sourcing.",
      "3": "I have a good founder network.",
      "4": "I have excellent founder access.",
      "5": "I have legendary founder discovery."
    },
    D7: {
      "1": "I am disconnected and dismissive.",
      "2": "I listen but do not learn.",
      "3": "I am open to learning from founders.",
      "4": "I learn well from my portfolio.",
      "5": "I am a legend for learning."
    },
    D8: {
      "1": "I disappear on loss.",
      "2": "I offer limited support.",
      "3": "I offer some support.",
      "4": "I offer good support through rejection.",
      "5": "I am a legend for crisis support."
    },
    D9: {
      "1": "I miss founder edges.",
      "2": "I acknowledge edges but do not value them.",
      "3": "I value the founder's unique edge.",
      "4": "I bet on the founder's edge.",
      "5": "I am a legendary edge spotter."
    },
    EiR1: {
      "1": "I always pick the \"safe\" profile (often wrong).",
      "2": "I usually pick well but have some bias.",
      "3": "I have mixed results; I am learning.",
      "4": "I am learning from my failures.",
      "5": "I am a legend for prediction."
    },
    EiR2: {
      "1": "I manage all risks (blind confidence).",
      "2": "I miss some risks; I am aware.",
      "3": "I miss several risks.",
      "4": "I am blind to portfolio risks.",
      "5": "I manage my portfolio scientifically."
    },
    EiR3: {
      "1": "I let excellent relations deteriorate.",
      "2": "I maintain good relations mostly.",
      "3": "I manage some tension.",
      "4": "I see significant tension.",
      "5": "I am a legend at navigating dynamics."
    },
    EiR4: {
      "1": "I ignore conflicts.",
      "2": "I have minimal conflicts.",
      "3": "I have some portfolio conflicts.",
      "4": "I have significant conflicts.",
      "5": "I manage portfolio conflicts fairly."
    },
    EiR5: {
      "1": "I hedge risks poorly.",
      "2": "I am mostly safe; I have some risk.",
      "3": "I have some economic risk exposure.",
      "4": "I have significant economic risk.",
      "5": "I have mapped all macro risks."
    },
    EiR6: {
      "1": "I ignore regulatory risks.",
      "2": "I see minimal risks; manageable.",
      "3": "I see some regulatory risks.",
      "4": "I see major regulatory risks.",
      "5": "I am a regulatory expert."
    },
    EiR7: {
      "1": "I allocate optimally (overconfidence).",
      "2": "I have good allocation; some risk.",
      "3": "I have reasonable allocation.",
      "4": "I have questionable allocation.",
      "5": "I allocate poorly (risk state)."
    },
    EiR8: {
      "1": "My portfolio is fresh (ignorance).",
      "2": "My portfolio is mostly fresh.",
      "3": "I see some fatigue; I manage it.",
      "4": "I see significant fatigue.",
      "5": "My portfolio is severely burned out."
    },
    EiR9: {
      "1": "I back for pure merit.",
      "2": "I back mostly for merit.",
      "3": "I have some agenda mixed in.",
      "4": "I have a significant agenda.",
      "5": "I back for pure agenda/politics (risk state)."
    }
  },
  L2: {
    D1: {
      "1": "I cannot assess quality.",
      "2": "I have a vague sense of MVP quality.",
      "3": "I can assess MVPs reasonably well.",
      "4": "I am an expert at MVP quality.",
      "5": "I am a legendary MVP judge."
    },
    D2: {
      "1": "I cannot spot risks.",
      "2": "I have a vague sense of risks.",
      "3": "I can spot major risks.",
      "4": "I am an expert at risk spotting.",
      "5": "I am a legend at identifying technical risk."
    },
    D3: {
      "1": "My portfolio has failed POCs.",
      "2": "My portfolio has weak POCs.",
      "3": "My portfolio has good POC quality.",
      "4": "My portfolio has strong POCs.",
      "5": "My portfolio produces exceptional quality."
    },
    D4: {
      "1": "My founders ignore feedback.",
      "2": "My founders get minimal feedback.",
      "3": "My founders get good feedback.",
      "4": "My founders have strong feedback loops.",
      "5": "My portfolio is customer-obsessed."
    },
    D5: {
      "1": "My portfolio builds wastefully.",
      "2": "My portfolio has some waste.",
      "3": "My portfolio has good efficiency.",
      "4": "My portfolio has excellent efficiency.",
      "5": "My portfolio demonstrates exceptional efficiency."
    },
    D6: {
      "1": "My portfolio has no unique angle.",
      "2": "My portfolio has an emerging angle.",
      "3": "My portfolio has a clear unique angle.",
      "4": "My portfolio has a strong unique angle.",
      "5": "My portfolio has exceptional angles."
    },
    D7: {
      "1": "My portfolio is not learning.",
      "2": "My portfolio has minimal learning.",
      "3": "My portfolio is growing in knowledge.",
      "4": "My portfolio has good learning.",
      "5": "My portfolio engages in deep learning."
    },
    D8: {
      "1": "My portfolio teams are weak.",
      "2": "My portfolio teams are moderate.",
      "3": "My portfolio teams are good.",
      "4": "My portfolio teams are strong.",
      "5": "My portfolio teams are exceptional."
    },
    D9: {
      "1": "My portfolio has no velocity.",
      "2": "My portfolio has slow velocity.",
      "3": "My portfolio has good velocity.",
      "4": "My portfolio has high velocity.",
      "5": "My portfolio has exceptional velocity."
    },
    EiR1: {
      "1": "I assume it works; untested.",
      "2": "I do minimal testing.",
      "3": "I ensure some testing is done.",
      "4": "I ensure thorough testing.",
      "5": "I require scientific proof of feasibility."
    },
    EiR2: {
      "1": "I am blind to all debt.",
      "2": "I miss major debt.",
      "3": "I know most debt risks.",
      "4": "I know key debt risks.",
      "5": "I have identified all debt."
    },
    EiR3: {
      "1": "My teams are weak.",
      "2": "My teams have moderate gaps.",
      "3": "My teams are good.",
      "4": "My teams are strong.",
      "5": "My teams are exceptional."
    },
    EiR4: {
      "1": "It is already obsolete.",
      "2": "There is a risk of obsolescence.",
      "3": "It has moderate risk.",
      "4": "It has low risk.",
      "5": "It is future-proof."
    },
    EiR5: {
      "1": "It is not sustainable.",
      "2": "It is fragile.",
      "3": "It is sustainable.",
      "4": "It is highly sustainable.",
      "5": "It is perpetual."
    },
    EiR6: {
      "1": "There is no path; chaos.",
      "2": "There is a vague path.",
      "3": "There is a clear path.",
      "4": "There is a strong path.",
      "5": "There is perfect visibility."
    },
    EiR7: {
      "1": "I make no investment.",
      "2": "I make minimal investment.",
      "3": "I make good investment.",
      "4": "I make strong investment.",
      "5": "I make exceptional investment."
    },
    EiR8: {
      "1": "My teams have severe burnout.",
      "2": "My teams have significant fatigue.",
      "3": "My teams have some fatigue.",
      "4": "My teams are managing well.",
      "5": "My teams are fresh and energized."
    },
    EiR9: {
      "1": "I fund pure ego/hype.",
      "2": "I fund mostly ego; some substance.",
      "3": "I fund mixed motives.",
      "4": "I fund mostly mission; some ego.",
      "5": "I fund pure, mission-driven builds."
    }
  },
  L3: {
    D1: {
      "1": "I cannot identify traction.",
      "2": "I have a vague sense of traction.",
      "3": "I can identify traction signals.",
      "4": "I am an expert traction spotter.",
      "5": "I am a legendary traction finder."
    },
    D2: {
      "1": "My portfolio has no paying users.",
      "2": "My portfolio has few paying users.",
      "3": "My portfolio has some paying users.",
      "4": "My portfolio has a good paying base.",
      "5": "My portfolio has a strong paying base."
    },
    D3: {
      "1": "My portfolio churns users.",
      "2": "My portfolio has poor retention.",
      "3": "My portfolio has moderate retention.",
      "4": "My portfolio has good retention.",
      "5": "My portfolio has excellent retention."
    },
    D4: {
      "1": "My portfolio has no organic growth.",
      "2": "My portfolio has minimal organic growth.",
      "3": "My portfolio has some organic growth.",
      "4": "My portfolio has good organic growth.",
      "5": "My portfolio has viral, organic growth."
    },
    D5: {
      "1": "My portfolio loses money on sales.",
      "2": "My portfolio has poor economics.",
      "3": "My portfolio is breaking even.",
      "4": "My portfolio has good margins.",
      "5": "My portfolio has excellent economics."
    },
    D6: {
      "1": "My portfolio has no differentiation.",
      "2": "My portfolio has slight differentiation.",
      "3": "My portfolio has clear differences.",
      "4": "My portfolio has strong differentiation.",
      "5": "My portfolio has unique positioning."
    },
    D7: {
      "1": "My portfolio has no momentum.",
      "2": "My portfolio has slow momentum.",
      "3": "My portfolio is building momentum.",
      "4": "My portfolio has strong momentum.",
      "5": "My portfolio has viral momentum."
    },
    D8: {
      "1": "My portfolio has no moat.",
      "2": "My portfolio has a weak moat.",
      "3": "My portfolio has a reasonable moat.",
      "4": "My portfolio has a good moat.",
      "5": "My portfolio is strongly defensible."
    },
    D9: {
      "1": "I have no market leaders.",
      "2": "I have marginal market positions.",
      "3": "I have emerging leaders.",
      "4": "I have clear leaders.",
      "5": "I back the market leaders."
    },
    EiR1: {
      "1": "They are assumed and untested.",
      "2": "I do minimal testing.",
      "3": "I do some testing.",
      "4": "I do thorough testing.",
      "5": "They are scientifically validated."
    },
    EiR2: {
      "1": "I am blind to threats.",
      "2": "I underestimate threats.",
      "3": "I am aware of threats.",
      "4": "I have deep competitive awareness.",
      "5": "I have expert mapping of threats."
    },
    EiR3: {
      "1": "Teams are falling apart.",
      "2": "There is significant tension.",
      "3": "There is some tension.",
      "4": "Teams have good cohesion.",
      "5": "There are legend builders."
    },
    EiR4: {
      "1": "They are easily copied.",
      "2": "There is high copy risk.",
      "3": "There is moderate risk.",
      "4": "There is low copy risk.",
      "5": "They are hard to copy."
    },
    EiR5: {
      "1": "Markets are oversaturated.",
      "2": "Markets are highly saturated.",
      "3": "Markets have moderate saturation.",
      "4": "Markets are growing.",
      "5": "Markets are emerging."
    },
    EiR6: {
      "1": "They are non-compliant.",
      "2": "They have major violations.",
      "3": "They have some issues.",
      "4": "They are mostly compliant.",
      "5": "They are fully compliant."
    },
    EiR7: {
      "1": "It is not sustainable.",
      "2": "It is fragile.",
      "3": "It is somewhat sustainable.",
      "4": "It is sustainable.",
      "5": "It is highly sustainable."
    },
    EiR8: {
      "1": "Teams are burned out.",
      "2": "Teams have significant fatigue.",
      "3": "Teams have some fatigue.",
      "4": "Teams are managing well.",
      "5": "Teams are fresh and energized."
    },
    EiR9: {
      "1": "I back artificial traction.",
      "2": "I back mostly hype/fake.",
      "3": "I back mixed real/fake traction.",
      "4": "I back mostly organic traction.",
      "5": "I back pure, organic traction."
    }
  },
  L4: {
    D1: {
      "1": "I cannot identify momentum.",
      "2": "I see vague growth signals.",
      "3": "I can identify growth signals.",
      "4": "I am an expert growth spotter.",
      "5": "I am a legendary growth finder."
    },
    D2: {
      "1": "My portfolio has no growth.",
      "2": "My portfolio has slow growth.",
      "3": "My portfolio has moderate growth.",
      "4": "My portfolio is growing fast.",
      "5": "My portfolio shows exponential growth."
    },
    D3: {
      "1": "My portfolio has no process.",
      "2": "My portfolio is somewhat repeatable.",
      "3": "My portfolio has a repeatable process.",
      "4": "My portfolio has a refined process.",
      "5": "My portfolio uses predictable engines."
    },
    D4: {
      "1": "My portfolio cannot expand.",
      "2": "My portfolio has slow expansion.",
      "3": "My portfolio is expanding to new markets.",
      "4": "My portfolio has strong expansion.",
      "5": "My portfolio achieves exceptional expansion."
    },
    D5: {
      "1": "My portfolio's economics get worse.",
      "2": "My portfolio sees margin pressure.",
      "3": "My portfolio is good at scale.",
      "4": "My portfolio improves at scale.",
      "5": "My portfolio's economics become exceptional."
    },
    D6: {
      "1": "My portfolio has no moat.",
      "2": "My portfolio has a weak moat.",
      "3": "My portfolio has a growing moat.",
      "4": "My portfolio has a strong moat.",
      "5": "My portfolio has legendary moats."
    },
    D7: {
      "1": "My portfolio has no leadership.",
      "2": "My portfolio is an emerging leader.",
      "3": "My portfolio is a clear leader.",
      "4": "My portfolio is a strong leader.",
      "5": "My portfolio contains the market leaders."
    },
    D8: {
      "1": "My portfolio's culture has collapsed.",
      "2": "My portfolio's culture is diluted.",
      "3": "My portfolio's culture is maintained.",
      "4": "My portfolio's culture is strengthened.",
      "5": "My portfolio has legendary culture."
    },
    D9: {
      "1": "My portfolio has no leaders.",
      "2": "My portfolio has emerging leaders.",
      "3": "My portfolio has clear leaders.",
      "4": "My portfolio has strong leaders.",
      "5": "My portfolio contains industry icons."
    },
    EiR1: {
      "1": "They are assumed and untested.",
      "2": "I do minimal testing.",
      "3": "I do some testing.",
      "4": "I do thorough testing.",
      "5": "They are scientifically proven."
    },
    EiR2: {
      "1": "I see no limits (blind).",
      "2": "I face unknown ceilings.",
      "3": "I know the limits.",
      "4": "I plan past the ceiling.",
      "5": "I see unlimited addressable markets."
    },
    EiR3: {
      "1": "They cannot hire.",
      "2": "Hiring is lagging.",
      "3": "They are keeping pace.",
      "4": "Hiring is ahead of growth.",
      "5": "They attract legendary talent."
    },
    EiR4: {
      "1": "Competitors are dormant.",
      "2": "There are some competitors.",
      "3": "Competitors are active.",
      "4": "Competition is intense.",
      "5": "I face an existential threat."
    },
    EiR5: {
      "1": "I believe they thrive (overconfidence).",
      "2": "They are mostly resilient.",
      "3": "There is some impact.",
      "4": "There is significant impact.",
      "5": "They would die in a recession."
    },
    EiR6: {
      "1": "They are blocked internationally.",
      "2": "There is minimal international presence.",
      "3": "They have some international presence.",
      "4": "They have good international growth.",
      "5": "They are globally ready."
    },
    EiR7: {
      "1": "They will never profit.",
      "2": "The path is unclear.",
      "3": "They have a clear path.",
      "4": "They are approaching profit.",
      "5": "They are already profitable."
    },
    EiR8: {
      "1": "Teams are thriving (ignorance).",
      "2": "Teams are mostly fresh.",
      "3": "There is some strain.",
      "4": "There is significant strain.",
      "5": "Teams are breaking."
    },
    EiR9: {
      "1": "I back pure mission focus.",
      "2": "I back mostly mission.",
      "3": "I have mixed motivations.",
      "4": "I back some ego/money.",
      "5": "I back pure exits/returns."
    }
  },
  L5: {
    D1: {
      "1": "I cannot assess margins.",
      "2": "I have a vague sense of profit.",
      "3": "I can identify profitability.",
      "4": "I am an expert profit spotter.",
      "5": "I am a legendary profit finder."
    },
    D2: {
      "1": "I see no leverage.",
      "2": "I see minimal leverage.",
      "3": "I see good leverage.",
      "4": "I see strong leverage.",
      "5": "I see exceptional leverage."
    },
    D3: {
      "1": "My portfolio has low LTV.",
      "2": "My portfolio has low LTV.",
      "3": "My portfolio has good LTV.",
      "4": "My portfolio has high LTV.",
      "5": "My portfolio has exceptional LTV."
    },
    D4: {
      "1": "My portfolio has high CAC.",
      "2": "My portfolio has high CAC.",
      "3": "My portfolio has healthy CAC.",
      "4": "My portfolio has low CAC.",
      "5": "My portfolio has exceptional CAC efficiency."
    },
    D5: {
      "1": "My portfolio has high churn.",
      "2": "My portfolio has moderate churn.",
      "3": "My portfolio has low churn.",
      "4": "My portfolio has very low churn.",
      "5": "My portfolio has near-zero churn."
    },
    D6: {
      "1": "My portfolio is cash-negative.",
      "2": "My portfolio is break-even.",
      "3": "My portfolio is cash-positive.",
      "4": "My portfolio has strong cash.",
      "5": "My portfolio is cash-rich."
    },
    D7: {
      "1": "My portfolio has no strategy.",
      "2": "My portfolio has basic planning.",
      "3": "My portfolio has a clear strategy.",
      "4": "My portfolio is optimized.",
      "5": "My portfolio uses expert optimization."
    },
    D8: {
      "1": "My portfolio allocates wastefully.",
      "2": "My portfolio is inefficient.",
      "3": "My portfolio allocates well.",
      "4": "My portfolio allocates efficiently.",
      "5": "My portfolio allocates exceptionally."
    },
    D9: {
      "1": "My portfolio wastes capital.",
      "2": "My portfolio has minimal reinvestment.",
      "3": "My portfolio has a good strategy.",
      "4": "My portfolio has strong reinvestment.",
      "5": "My portfolio has exceptional reinvestment."
    },
    EiR1: {
      "1": "They are assumed and untested.",
      "2": "I do minimal testing.",
      "3": "I do some testing.",
      "4": "I do thorough testing.",
      "5": "They are scientifically proven."
    },
    EiR2: {
      "1": "I assume unlimited profit.",
      "2": "I face unknown risks.",
      "3": "I identify clear risks.",
      "4": "I plan past risks.",
      "5": "I face unknown unknowns."
    },
    EiR3: {
      "1": "Compensation is unsustainable.",
      "2": "Costs are rising.",
      "3": "Compensation is managed.",
      "4": "I have talent cost leverage.",
      "5": "Efficiency is exceptional."
    },
    EiR4: {
      "1": "Prices are rising (advantage).",
      "2": "Prices are stable.",
      "3": "There is some pressure.",
      "4": "There is significant pressure.",
      "5": "Pricing is under siege."
    },
    EiR5: {
      "1": "High fixed costs.",
      "2": "Mixed costs.",
      "3": "Balanced costs.",
      "4": "Variable-heavy costs.",
      "5": "Exceptional flexibility."
    },
    EiR6: {
      "1": "Returns are unrealistic.",
      "2": "Returns are optimistic.",
      "3": "Returns are realistic.",
      "4": "Returns are conservative.",
      "5": "Returns are exceeding expectations."
    },
    EiR7: {
      "1": "Financials are hidden.",
      "2": "Visibility is limited.",
      "3": "Financials are clear.",
      "4": "Visibility is excellent.",
      "5": "Financials are completely transparent."
    },
    EiR8: {
      "1": "They have a growth-only obsession.",
      "2": "They prioritize growth over profit.",
      "3": "They have a balanced mindset.",
      "4": "They are profit-focused.",
      "5": "They have a profit obsession."
    },
    EiR9: {
      "1": "I am driven by pure greed.",
      "2": "I am mostly greed-driven.",
      "3": "I have mixed motives.",
      "4": "I am mostly mission-driven.",
      "5": "I am driven by pure, mission-focused intent."
    }
  },
  L6: {
    D1: {
      "1": "I cannot identify vision.",
      "2": "I have a vague sense of vision.",
      "3": "I can identify vision.",
      "4": "I am an expert vision spotter.",
      "5": "I am a legendary visionary spotter."
    },
    D2: {
      "1": "My portfolio has no pipeline.",
      "2": "My portfolio has a thin pipeline.",
      "3": "My portfolio has a growing pipeline.",
      "4": "My portfolio has a strong pipeline.",
      "5": "My portfolio has an exceptional pipeline."
    },
    D3: {
      "1": "My portfolio has bad strategy.",
      "2": "My portfolio has weak strategy.",
      "3": "My portfolio has good strategy.",
      "4": "My portfolio has strong strategy.",
      "5": "My portfolio has exceptional strategy."
    },
    D4: {
      "1": "My portfolio cannot drive innovation.",
      "2": "My portfolio has minimal innovation.",
      "3": "My portfolio has good innovation.",
      "4": "My portfolio has strong innovation.",
      "5": "My portfolio is legendary for innovation."
    },
    D5: {
      "1": "My portfolio has chaotic scaling.",
      "2": "My portfolio struggles to scale.",
      "3": "My portfolio has good scaling.",
      "4": "My portfolio scales smoothly.",
      "5": "My portfolio has exceptional scaling."
    },
    D6: {
      "1": "My portfolio has low trust.",
      "2": "My portfolio has low trust.",
      "3": "My portfolio has good trust.",
      "4": "My portfolio has high trust.",
      "5": "My portfolio has exceptional trust."
    },
    D7: {
      "1": "My portfolio attracts low-quality talent.",
      "2": "My portfolio attracts moderate talent.",
      "3": "My portfolio attracts good talent.",
      "4": "My portfolio attracts strong talent.",
      "5": "My portfolio attracts exceptional talent."
    },
    D8: {
      "1": "My portfolio has no mission.",
      "2": "My portfolio has a vague mission.",
      "3": "My portfolio has a clear mission.",
      "4": "My portfolio has a strong mission.",
      "5": "My portfolio has exceptional mission."
    },
    D9: {
      "1": "My portfolio has no influence.",
      "2": "My portfolio has emerging influence.",
      "3": "My portfolio has good influence.",
      "4": "My portfolio has strong influence.",
      "5": "My portfolio has exceptional influence."
    },
    EiR1: {
      "1": "They are assumed and untested.",
      "2": "I do minimal testing.",
      "3": "I do some testing.",
      "4": "I do thorough testing.",
      "5": "They are scientifically proven."
    },
    EiR2: {
      "1": "I am completely blind.",
      "2": "I miss major risks.",
      "3": "I know most risks.",
      "4": "I know key risks.",
      "5": "I am an expert finder."
    },
    EiR3: {
      "1": "They have weak teams.",
      "2": "They have significant gaps.",
      "3": "They have good teams.",
      "4": "They have strong teams.",
      "5": "They have exceptional teams."
    },
    EiR4: {
      "1": "It is rapidly outdated.",
      "2": "There is risk of obsolescence.",
      "3": "It has good relevance.",
      "4": "It has strong relevance.",
      "5": "It is always relevant."
    },
    EiR5: {
      "1": "It is short-term.",
      "2": "It is fragile.",
      "3": "It has good longevity.",
      "4": "It is long-term.",
      "5": "It is generational."
    },
    EiR6: {
      "1": "There is no plan; risky.",
      "2": "The plan is vague.",
      "3": "There is a good plan.",
      "4": "There is a strong plan.",
      "5": "There is an exceptional plan."
    },
    EiR7: {
      "1": "There are no resources.",
      "2": "Investment is minimal.",
      "3": "There are good resources.",
      "4": "There are strong resources.",
      "5": "There are exceptional resources."
    },
    EiR8: {
      "1": "Leaders have severe burnout.",
      "2": "Leaders have significant fatigue.",
      "3": "Leaders have some fatigue.",
      "4": "Leaders are managing well.",
      "5": "Leaders are fresh and energized."
    },
    EiR9: {
      "1": "I invest for financial reasons only.",
      "2": "I invest mostly for power.",
      "3": "I have mixed motives.",
      "4": "I invest mostly for mission.",
      "5": "I invest for mission-driven reasons."
    }
  },
  L7: {
    D1: {
      "1": "I cannot spot unicorns.",
      "2": "I have a vague sense of unicorns.",
      "3": "I can identify unicorns.",
      "4": "I am an expert unicorn spotter.",
      "5": "I am a legendary unicorn finder."
    },
    D2: {
      "1": "My portfolio has no legacy.",
      "2": "My portfolio has minimal legacy.",
      "3": "My portfolio has a clear legacy.",
      "4": "My portfolio has a strong legacy.",
      "5": "My portfolio creates generational impact."
    },
    D3: {
      "1": "My portfolio has no metrics.",
      "2": "My portfolio has low metrics.",
      "3": "My portfolio has good metrics.",
      "4": "My portfolio has strong metrics.",
      "5": "My portfolio has exceptional metrics."
    },
    D4: {
      "1": "My portfolio has no inspiration.",
      "2": "My portfolio has minimal inspiration.",
      "3": "My portfolio is inspiring.",
      "4": "My portfolio is highly inspiring.",
      "5": "My portfolio provides exceptional inspiration."
    },
    D5: {
      "1": "My portfolio has no global reach.",
      "2": "My portfolio has limited global reach.",
      "3": "My portfolio has international presence.",
      "4": "My portfolio has strong global reach.",
      "5": "My portfolio has exceptional global reach."
    },
    D6: {
      "1": "My portfolio has no disruption.",
      "2": "My portfolio has incremental innovation.",
      "3": "My portfolio has clear innovation.",
      "4": "My portfolio has strong disruption.",
      "5": "My portfolio is revolutionary."
    },
    D7: {
      "1": "My portfolio has no leadership.",
      "2": "My portfolio has emerging leadership.",
      "3": "My portfolio has clear leadership.",
      "4": "My portfolio has strong leadership.",
      "5": "My portfolio provides legendary leadership."
    },
    D8: {
      "1": "My portfolio has no ecosystem.",
      "2": "My portfolio has a small ecosystem.",
      "3": "My portfolio has a good ecosystem.",
      "4": "My portfolio has a strong ecosystem.",
      "5": "My portfolio has an exceptional ecosystem."
    },
    D9: {
      "1": "My portfolio creates no change.",
      "2": "My portfolio creates limited change.",
      "3": "My portfolio creates some change.",
      "4": "My portfolio creates strong change.",
      "5": "My portfolio creates permanent change."
    },
    EiR1: {
      "1": "They are assumed and untested.",
      "2": "I do minimal testing.",
      "3": "I do some testing.",
      "4": "I do thorough testing.",
      "5": "They are scientifically proven."
    },
    EiR2: {
      "1": "I am completely blind.",
      "2": "I miss major side effects.",
      "3": "I know most risks.",
      "4": "I know key risks.",
      "5": "I have expert identification."
    },
    EiR3: {
      "1": "The ecosystem is unhealthy.",
      "2": "There is some risk.",
      "3": "The ecosystem has good health.",
      "4": "The ecosystem has strong health.",
      "5": "The ecosystem has exceptional health."
    },
    EiR4: {
      "1": "It cannot survive.",
      "2": "It has some durability.",
      "3": "It has good durability.",
      "4": "It has strong durability.",
      "5": "It is eternal."
    },
    EiR5: {
      "1": "It cannot sustain growth.",
      "2": "Growth is fragile.",
      "3": "It has sustainable growth.",
      "4": "It is highly sustainable.",
      "5": "It is perpetual."
    },
    EiR6: {
      "1": "There is unproven value.",
      "2": "Value is uncertain.",
      "3": "There is clear value.",
      "4": "There is strong value.",
      "5": "There is exceptional value."
    },
    EiR7: {
      "1": "There are no metrics.",
      "2": "Metrics are vague.",
      "3": "There are clear metrics.",
      "4": "There are strong metrics.",
      "5": "There are exceptional metrics."
    },
    EiR8: {
      "1": "There is severe exhaustion.",
      "2": "There is significant fatigue.",
      "3": "There is some fatigue.",
      "4": "Teams are managing well.",
      "5": "Teams are fresh and energized."
    },
    EiR9: {
      "1": "I back for pure exits.",
      "2": "I back mostly for ego.",
      "3": "I have mixed motives.",
      "4": "I back mostly for impact.",
      "5": "I back for pure impact."
    }
  },
  L8: {
    D1: {
      "1": "I cannot identify stewards.",
      "2": "I have a vague sense of stewardship.",
      "3": "I can identify stewards.",
      "4": "I am an expert steward spotter.",
      "5": "I am a legendary finder of stewards."
    },
    D2: {
      "1": "My portfolio cares for shareholders only.",
      "2": "My portfolio cares for some stakeholders.",
      "3": "My portfolio balances most stakeholders.",
      "4": "My portfolio cares for all stakeholders.",
      "5": "My portfolio provides exceptional care."
    },
    D3: {
      "1": "My portfolio has no ESG focus.",
      "2": "My portfolio has limited ESG.",
      "3": "My portfolio has good ESG practices.",
      "4": "My portfolio has strong ESG focus.",
      "5": "My portfolio has exceptional ESG focus."
    },
    D4: {
      "1": "My portfolio has no social impact.",
      "2": "My portfolio has limited impact.",
      "3": "My portfolio has good impact.",
      "4": "My portfolio has strong impact.",
      "5": "My portfolio has exceptional impact."
    },
    D5: {
      "1": "My portfolio is opaque.",
      "2": "My portfolio has limited transparency.",
      "3": "My portfolio has good governance.",
      "4": "My portfolio has strong governance.",
      "5": "My portfolio has exceptional governance."
    },
    D6: {
      "1": "My portfolio has no ethics.",
      "2": "My portfolio has limited ethics.",
      "3": "My portfolio has good ethics.",
      "4": "My portfolio has strong ethics.",
      "5": "My portfolio has exceptional ethics."
    },
    D7: {
      "1": "My portfolio has no community.",
      "2": "My portfolio has limited community.",
      "3": "My portfolio has good partnerships.",
      "4": "My portfolio has strong partnerships.",
      "5": "My portfolio has exceptional partnerships."
    },
    D8: {
      "1": "My portfolio has quarterly focus.",
      "2": "My portfolio has annual focus.",
      "3": "My portfolio has 5–10 year vision.",
      "4": "My portfolio has 20+ year vision.",
      "5": "My portfolio is generational."
    },
    D9: {
      "1": "My portfolio has no values.",
      "2": "My portfolio has limited alignment.",
      "3": "My portfolio has good alignment.",
      "4": "My portfolio has strong alignment.",
      "5": "My portfolio has exceptional alignment."
    },
    EiR1: {
      "1": "They are assumed and untested.",
      "2": "I do minimal testing.",
      "3": "I do some testing.",
      "4": "I do thorough testing.",
      "5": "They are scientifically proven."
    },
    EiR2: {
      "1": "I am completely blind.",
      "2": "I miss major risks.",
      "3": "I know most risks.",
      "4": "I know key risks.",
      "5": "I have expert identification."
    },
    EiR3: {
      "1": "It is unhealthy.",
      "2": "There are some conflicts.",
      "3": "It has good health.",
      "4": "It has strong harmony.",
      "5": "It has exceptional health."
    },
    EiR4: {
      "1": "It is completely inauthentic.",
      "2": "It is mostly inauthentic.",
      "3": "It is mostly authentic.",
      "4": "It is highly authentic.",
      "5": "It is exceptional and authentic."
    },
    EiR5: {
      "1": "There are no metrics.",
      "2": "Metrics are vague.",
      "3": "There are good metrics.",
      "4": "There are strong metrics.",
      "5": "There are exceptional metrics."
    },
    EiR6: {
      "1": "There is no accountability.",
      "2": "Mechanisms are limited.",
      "3": "There is good accountability.",
      "4": "There is strong accountability.",
      "5": "There is exceptional accountability."
    },
    EiR7: {
      "1": "It is mostly opaque.",
      "2": "It is mostly opaque.",
      "3": "It has good balance.",
      "4": "It has strong transparency.",
      "5": "It is exceptional."
    },
    EiR8: {
      "1": "There is burnout culture.",
      "2": "There is some tension.",
      "3": "There is good wellbeing.",
      "4": "There is strong culture.",
      "5": "There is exceptional wellbeing."
    },
    EiR9: {
      "1": "I am driven by pure extraction.",
      "2": "I am mostly extraction-driven.",
      "3": "I have mixed motives.",
      "4": "I am mostly stewardship-driven.",
      "5": "I am driven by pure stewardship."
    }
  }
};

let updateCount = 0;
let missingCount = 0;

for (const [levelId, dimensions] of Object.entries(allUpdates)) {
  if (!answerKeys[levelId]) {
    console.log(`Level ${levelId} not found`);
    continue;
  }
  
  for (const [dimId, grades] of Object.entries(dimensions)) {
    if (!answerKeys[levelId].questions?.[dimId]) {
      console.log(`${levelId} ${dimId} not found in answerKeys`);
      missingCount++;
      continue;
    }
    
    const stakeholderData = answerKeys[levelId].questions[dimId].stakeholders?.[STAKEHOLDER];
    if (!stakeholderData) {
      console.log(`${levelId} ${dimId} - Stakeholder ${STAKEHOLDER} not found`);
      missingCount++;
      continue;
    }
    
    for (const [grade, answer] of Object.entries(grades)) {
      stakeholderData.grades[grade] = answer;
    }
    updateCount++;
    
    console.log(`Updated ${levelId} ${dimId} (5 grades)`);
  }
}

fs.writeFileSync(answerKeysPath, JSON.stringify(answerKeys, null, 2), 'utf-8');

console.log(`\n✅ Updated ${updateCount} Investor entries with all 5 grades.`);
console.log(`⚠️ Missing entries: ${missingCount}`);
console.log(`Total individual grade updates: ${updateCount * 5}`);
console.log(`File saved to: ${answerKeysPath}`);
