import type { Express } from "express";
import { createServer, type Server } from "http";
import { storage, db } from "./storage";
import { insertAssessmentSchema, insertLeaderboardEntrySchema, insertUserAssessmentProgressSchema, insertLevelCompletionSchema, insertFreeUserSchema, conversations, messages, userSuggestions, users, freeUsers, userSessions, userIdeas, ideaRoles, assessmentSessions, valueJourneyAssessments, contestCases, type PauseState } from "@shared/schema";
import { fromError } from "zod-validation-error";
import { eq, desc, and, sql } from "drizzle-orm";
import OpenAI from "openai";

const openai = new OpenAI({
  apiKey: process.env.AI_INTEGRATIONS_OPENAI_API_KEY,
  baseURL: process.env.AI_INTEGRATIONS_OPENAI_BASE_URL,
});

export async function registerRoutes(
  httpServer: Server,
  app: Express
): Promise<Server> {
  app.post("/api/assessments", async (req, res) => {
    try {
      const validatedData = insertAssessmentSchema.parse(req.body);
      const assessment = await storage.createAssessment(validatedData);
      res.status(201).json({ success: true, assessment });
    } catch (error: any) {
      if (error.name === 'ZodError') {
        const validationError = fromError(error);
        res.status(400).json({ 
          success: false, 
          error: validationError.toString() 
        });
      } else {
        console.error("Error creating assessment:", error);
        res.status(500).json({ 
          success: false, 
          error: "Failed to save assessment" 
        });
      }
    }
  });

  app.get("/api/assessments/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        res.status(400).json({ success: false, error: "Invalid assessment ID" });
        return;
      }
      
      const assessment = await storage.getAssessmentById(id);
      if (!assessment) {
        res.status(404).json({ success: false, error: "Assessment not found" });
        return;
      }
      
      res.json({ success: true, assessment });
    } catch (error) {
      console.error("Error fetching assessment:", error);
      res.status(500).json({ 
        success: false, 
        error: "Failed to fetch assessment" 
      });
    }
  });

  // Leaderboard endpoints
  app.get("/api/leaderboard", async (req, res) => {
    try {
      const limit = req.query.limit ? parseInt(req.query.limit as string) : 50;
      const phase = req.query.phase ? parseInt(req.query.phase as string) : undefined;
      
      const entries = await storage.getLeaderboardEntries(limit, phase);
      res.json({ success: true, entries });
    } catch (error) {
      console.error("Error fetching leaderboard:", error);
      res.status(500).json({ 
        success: false, 
        error: "Failed to fetch leaderboard" 
      });
    }
  });

  app.post("/api/leaderboard", async (req, res) => {
    try {
      const validatedData = insertLeaderboardEntrySchema.parse(req.body);
      const entry = await storage.createLeaderboardEntry(validatedData);
      res.status(201).json({ success: true, entry });
    } catch (error: any) {
      if (error.name === 'ZodError') {
        const validationError = fromError(error);
        res.status(400).json({ 
          success: false, 
          error: validationError.toString() 
        });
      } else {
        console.error("Error creating leaderboard entry:", error);
        res.status(500).json({ 
          success: false, 
          error: "Failed to save to leaderboard" 
        });
      }
    }
  });

  // AI Coaching Chatbot endpoints
  
  // Get all coaching conversations
  app.get("/api/coaching/conversations", async (req, res) => {
    try {
      const allConversations = await db.select().from(conversations).orderBy(desc(conversations.createdAt));
      res.json({ success: true, conversations: allConversations });
    } catch (error) {
      console.error("Error fetching conversations:", error);
      res.status(500).json({ success: false, error: "Failed to fetch conversations" });
    }
  });

  // Create a new coaching conversation with assessment context
  app.post("/api/coaching/conversations", async (req, res) => {
    try {
      const { title, assessmentContext } = req.body;
      const [conversation] = await db.insert(conversations).values({
        title: title || "Coaching Session",
        assessmentContext: assessmentContext || null,
      }).returning();
      res.status(201).json({ success: true, conversation });
    } catch (error) {
      console.error("Error creating conversation:", error);
      res.status(500).json({ success: false, error: "Failed to create conversation" });
    }
  });

  // Get a single conversation with messages
  app.get("/api/coaching/conversations/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ success: false, error: "Invalid conversation ID" });
      }
      
      const [conversation] = await db.select().from(conversations).where(eq(conversations.id, id));
      if (!conversation) {
        return res.status(404).json({ success: false, error: "Conversation not found" });
      }
      
      const conversationMessages = await db.select().from(messages).where(eq(messages.conversationId, id)).orderBy(messages.createdAt);
      res.json({ success: true, conversation, messages: conversationMessages });
    } catch (error) {
      console.error("Error fetching conversation:", error);
      res.status(500).json({ success: false, error: "Failed to fetch conversation" });
    }
  });

  // Delete a conversation
  app.delete("/api/coaching/conversations/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      await db.delete(messages).where(eq(messages.conversationId, id));
      await db.delete(conversations).where(eq(conversations.id, id));
      res.status(204).send();
    } catch (error) {
      console.error("Error deleting conversation:", error);
      res.status(500).json({ success: false, error: "Failed to delete conversation" });
    }
  });

  // Send message and get AI coaching response (streaming)
  app.post("/api/coaching/conversations/:id/messages", async (req, res) => {
    try {
      const conversationId = parseInt(req.params.id);
      const { content } = req.body;

      if (!content) {
        return res.status(400).json({ success: false, error: "Message content is required" });
      }

      // Get conversation for context
      const [conversation] = await db.select().from(conversations).where(eq(conversations.id, conversationId));
      if (!conversation) {
        return res.status(404).json({ success: false, error: "Conversation not found" });
      }

      // Save user message
      await db.insert(messages).values({
        conversationId,
        role: "user",
        content,
      });

      // Get conversation history
      const conversationMessages = await db.select().from(messages).where(eq(messages.conversationId, conversationId)).orderBy(messages.createdAt);

      // Build system prompt with assessment context
      const assessmentContext = conversation.assessmentContext as any;
      let systemPrompt = `You are an expert startup coach and part of the i2u.ai (Ideas to Unicorns) coaching team. You provide personalized, actionable advice based on the user's Value Journey Quest assessment results.

Your coaching style:
- Be encouraging but honest - celebrate wins, acknowledge challenges
- Provide specific, actionable recommendations tailored to their journey level
- Reference successful unicorn stories and patterns when relevant
- Ask clarifying questions to understand their specific situation
- Focus on closing the gap between their current reality and aspirations
- Keep responses concise but valuable (2-3 paragraphs max unless they ask for more detail)
- Use the Gleams & Alicorns language when appropriate (Gleams = effort points, Alicorns = mastery multiplier)

`;

      if (assessmentContext) {
        systemPrompt += `
VALUE JOURNEY ASSESSMENT CONTEXT:
- Stakeholder Type: ${assessmentContext.stakeholder || 'Not specified'}
- Current Level: ${assessmentContext.currentLevel || 'L0'} (${assessmentContext.currentLevelName || 'Starting point'})
- Aspirational Level: ${assessmentContext.aspirationalLevel || 'L1'} (${assessmentContext.aspirationalLevelName || 'Next goal'})
- Current Reality Score: ${assessmentContext.currentScore || 0}%
- Aspirational Score: ${assessmentContext.aspirationalScore || 0}%
- Gap to Close: ${assessmentContext.gapScore || 0} points
- Gleams Earned: ${assessmentContext.gleamsEarned || 0}
- Alicorns: ${assessmentContext.alicornsEarned || 0}
${assessmentContext.isTrial ? '- Mode: Trial Run (exploring the platform)' : ''}

LEVEL CONTEXT:
L0: Conception (The Spark) - Resilience focus
L1: Initiation (The Hunt) - Validation focus  
L2: Formulation (The Build) - Feasibility focus
L3: Market Entry (The Launch) - Traction focus
L4: Scaling (The Rocket) - Growth focus
L5: Efficiency (The Optimization) - Profit focus
L6: Leadership (The Institution) - Innovation focus
L7: Unicorn (The Dominance) - Legacy focus
L8: Masters/Jedi (The Return) - Stewardship focus

Personalize your coaching based on their current level, aspirational level, and the gap between them. Help them understand what's needed to progress from their current state to their aspirations.
`;
      }

      const chatMessages: OpenAI.ChatCompletionMessageParam[] = [
        { role: "system", content: systemPrompt },
        ...conversationMessages.map((m) => ({
          role: m.role as "user" | "assistant",
          content: m.content,
        })),
      ];

      // Set up SSE
      res.setHeader("Content-Type", "text/event-stream");
      res.setHeader("Cache-Control", "no-cache");
      res.setHeader("Connection", "keep-alive");

      // Stream response from OpenAI
      const stream = await openai.chat.completions.create({
        model: "gpt-5",
        messages: chatMessages,
        stream: true,
        max_completion_tokens: 1024,
      });

      let fullResponse = "";

      for await (const chunk of stream) {
        const chunkContent = chunk.choices[0]?.delta?.content || "";
        if (chunkContent) {
          fullResponse += chunkContent;
          res.write(`data: ${JSON.stringify({ content: chunkContent })}\n\n`);
        }
      }

      // Save assistant message
      await db.insert(messages).values({
        conversationId,
        role: "assistant",
        content: fullResponse,
      });

      res.write(`data: ${JSON.stringify({ done: true })}\n\n`);
      res.end();
    } catch (error) {
      console.error("Error in coaching chat:", error);
      if (res.headersSent) {
        res.write(`data: ${JSON.stringify({ error: "Failed to get coaching response" })}\n\n`);
        res.end();
      } else {
        res.status(500).json({ success: false, error: "Failed to get coaching response" });
      }
    }
  });

  // i2u.ai External Registration API Integration
  const I2U_API_BASE = "https://global-leaderboard.i2u.ai";
  const I2U_API_KEY = process.env.I2U_API_KEY;

  // Check email availability
  app.get("/api/i2u/check-email", async (req, res) => {
    try {
      const email = req.query.email as string;
      if (!email) {
        return res.status(400).json({ success: false, error: "Email is required" });
      }

      const response = await fetch(`${I2U_API_BASE}/api/external/check-email?email=${encodeURIComponent(email)}`, {
        headers: {
          "Authorization": `Bearer ${I2U_API_KEY}`,
        },
      });

      const data = await response.json();
      res.json(data);
    } catch (error) {
      console.error("Error checking email:", error);
      res.status(500).json({ success: false, error: "Failed to check email availability" });
    }
  });

  // Full registration - create user directly
  app.post("/api/i2u/register", async (req, res) => {
    try {
      const { email, name, password, role, referralCode } = req.body;

      if (!email || !name || !password || !role) {
        return res.status(400).json({ 
          success: false, 
          error: "Email, name, password, and role are required" 
        });
      }

      const response = await fetch(`${I2U_API_BASE}/api/external/register`, {
        method: "POST",
        headers: {
          "Authorization": `Bearer ${I2U_API_KEY}`,
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          email,
          name,
          password,
          role,
          referralCode: referralCode || undefined,
        }),
      });

      const data = await response.json();
      
      if (!response.ok) {
        return res.status(response.status).json(data);
      }

      res.json(data);
    } catch (error) {
      console.error("Error registering user:", error);
      res.status(500).json({ success: false, error: "Failed to register user" });
    }
  });

  // Generate registration link with pre-filled data
  app.post("/api/i2u/registration-link", async (req, res) => {
    try {
      const { email, name, role, referralCode } = req.body;

      const response = await fetch(`${I2U_API_BASE}/api/external/registration-link`, {
        method: "POST",
        headers: {
          "Authorization": `Bearer ${I2U_API_KEY}`,
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          email,
          name,
          role,
          referralCode,
        }),
      });

      const data = await response.json();
      res.json(data);
    } catch (error) {
      console.error("Error generating registration link:", error);
      res.status(500).json({ success: false, error: "Failed to generate registration link" });
    }
  });

  // Value Journey Quest - User Progress Tracking
  
  // Get user's assessment progress (returns existing progress or null)
  app.get("/api/value-journey/progress/:userId", async (req, res) => {
    try {
      const { userId } = req.params;
      const progress = await storage.getUserProgress(userId);
      const completions = progress ? await storage.getLevelCompletions(userId) : [];
      
      res.json({ 
        success: true, 
        progress,
        completions,
        hasExistingProgress: !!progress
      });
    } catch (error) {
      console.error("Error fetching user progress:", error);
      res.status(500).json({ success: false, error: "Failed to fetch progress" });
    }
  });

  // Create initial progress with permanent stakeholder choice
  app.post("/api/value-journey/progress", async (req, res) => {
    try {
      const validatedData = insertUserAssessmentProgressSchema.parse(req.body);
      
      // Check if user already has progress
      const existing = await storage.getUserProgress(validatedData.userId);
      if (existing) {
        return res.status(400).json({ 
          success: false, 
          error: "User already has assessment progress. Stakeholder choice is permanent." 
        });
      }
      
      const progress = await storage.createUserProgress(validatedData);
      res.status(201).json({ success: true, progress });
    } catch (error: any) {
      if (error.name === 'ZodError') {
        const validationError = fromError(error);
        res.status(400).json({ success: false, error: validationError.toString() });
      } else {
        console.error("Error creating progress:", error);
        res.status(500).json({ success: false, error: "Failed to create progress" });
      }
    }
  });

  // Save level completion with IP address tracking
  app.post("/api/value-journey/completions", async (req, res) => {
    try {
      // Get client IP address
      const ipAddress = req.headers['x-forwarded-for'] as string 
        || req.socket.remoteAddress 
        || 'unknown';
      
      const completionData = {
        ...req.body,
        ipAddress: Array.isArray(ipAddress) ? ipAddress[0] : ipAddress.split(',')[0].trim()
      };
      
      const validatedData = insertLevelCompletionSchema.parse(completionData);
      const completion = await storage.createLevelCompletion(validatedData);
      
      // Update user's total gleams and alicorns
      const progress = await storage.getUserProgress(validatedData.userId);
      if (progress) {
        await storage.updateUserProgress(validatedData.userId, {
          totalGleams: progress.totalGleams + validatedData.gleamsEarned,
          totalAlicorns: String(parseFloat(progress.totalAlicorns) + parseFloat(validatedData.alicornsEarned)),
        });
      }
      
      res.status(201).json({ success: true, completion });
    } catch (error: any) {
      if (error.name === 'ZodError') {
        const validationError = fromError(error);
        res.status(400).json({ success: false, error: validationError.toString() });
      } else {
        console.error("Error saving completion:", error);
        res.status(500).json({ success: false, error: "Failed to save completion" });
      }
    }
  });

  // Update pause state
  app.put("/api/value-journey/progress/:userId/pause", async (req, res) => {
    try {
      const { userId } = req.params;
      const pauseState: PauseState | null = req.body.pauseState;
      
      await storage.updatePauseState(userId, pauseState);
      res.json({ success: true });
    } catch (error) {
      console.error("Error updating pause state:", error);
      res.status(500).json({ success: false, error: "Failed to update pause state" });
    }
  });

  // Request additional stakeholder
  app.post("/api/value-journey/progress/:userId/request-stakeholder", async (req, res) => {
    try {
      const { userId } = req.params;
      const { stakeholder } = req.body;
      
      if (!stakeholder) {
        return res.status(400).json({ success: false, error: "Stakeholder is required" });
      }
      
      await storage.requestAdditionalStakeholder(userId, stakeholder);
      res.json({ success: true, message: "Stakeholder request submitted" });
    } catch (error) {
      console.error("Error requesting stakeholder:", error);
      res.status(500).json({ success: false, error: "Failed to submit stakeholder request" });
    }
  });

  // ============ FREE USER ENDPOINTS ============
  
  // Get or create free user by session
  app.get("/api/free-user/:sessionId", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const user = await storage.getFreeUser(sessionId);
      
      if (user) {
        res.json({ success: true, user, exists: true });
      } else {
        res.json({ success: true, user: null, exists: false });
      }
    } catch (error) {
      console.error("Error fetching free user:", error);
      res.status(500).json({ success: false, error: "Failed to fetch free user" });
    }
  });

  // Create free user
  app.post("/api/free-user", async (req, res) => {
    try {
      const ipAddress = req.headers['x-forwarded-for'] as string || req.socket.remoteAddress || 'unknown';
      const userData = { ...req.body, ipAddress };
      
      const existingUser = await storage.getFreeUser(userData.sessionId);
      if (existingUser) {
        return res.json({ success: true, user: existingUser, created: false });
      }
      
      const user = await storage.createFreeUser(userData);
      res.status(201).json({ success: true, user, created: true });
    } catch (error: any) {
      console.error("Error creating free user:", error);
      res.status(500).json({ success: false, error: "Failed to create free user" });
    }
  });

  // Update free user
  app.put("/api/free-user/:sessionId", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const updates = req.body;
      
      const user = await storage.updateFreeUser(sessionId, updates);
      if (!user) {
        return res.status(404).json({ success: false, error: "Free user not found" });
      }
      
      res.json({ success: true, user });
    } catch (error) {
      console.error("Error updating free user:", error);
      res.status(500).json({ success: false, error: "Failed to update free user" });
    }
  });

  // Lock stakeholder for free user
  app.post("/api/free-user/:sessionId/lock-stakeholder", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const { stakeholder, displayName } = req.body;
      
      if (!stakeholder) {
        return res.status(400).json({ success: false, error: "Stakeholder is required" });
      }
      
      const user = await storage.updateFreeUser(sessionId, {
        primaryStakeholder: stakeholder,
        displayName: displayName || undefined,
        isStakeholderLocked: true,
      });
      
      if (!user) {
        return res.status(404).json({ success: false, error: "Free user not found" });
      }
      
      res.json({ success: true, user, stakeholderLocked: true });
    } catch (error) {
      console.error("Error locking stakeholder:", error);
      res.status(500).json({ success: false, error: "Failed to lock stakeholder" });
    }
  });

  // Add additional stakeholder role
  app.post("/api/free-user/:sessionId/add-stakeholder", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const { stakeholder } = req.body;
      
      if (!stakeholder) {
        return res.status(400).json({ success: false, error: "Stakeholder is required" });
      }
      
      const user = await storage.getFreeUser(sessionId);
      if (!user) {
        return res.status(404).json({ success: false, error: "Free user not found" });
      }
      
      const currentAdditional = user.additionalStakeholders || [];
      if (!currentAdditional.includes(stakeholder) && stakeholder !== user.primaryStakeholder) {
        const updatedUser = await storage.updateFreeUser(sessionId, {
          additionalStakeholders: [...currentAdditional, stakeholder],
        });
        res.json({ success: true, user: updatedUser, added: true });
      } else {
        res.json({ success: true, user, added: false, message: "Stakeholder already exists" });
      }
    } catch (error) {
      console.error("Error adding stakeholder:", error);
      res.status(500).json({ success: false, error: "Failed to add stakeholder" });
    }
  });

  // Save level selection and scores
  app.post("/api/free-user/:sessionId/save-levels", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const { currentLevel, aspirationalLevel, levelScores, completedLevels, totalGleams, totalAlicorns, currentAnswers, aspirationalAnswers } = req.body;
      
      const user = await storage.updateFreeUser(sessionId, {
        currentLevel,
        aspirationalLevel,
        levelScores,
        completedLevels,
        totalGleams,
        totalAlicorns: totalAlicorns?.toString() || "0",
        currentAnswers,
        aspirationalAnswers,
      });
      
      if (!user) {
        return res.status(404).json({ success: false, error: "Free user not found" });
      }
      
      res.json({ success: true, user });
    } catch (error) {
      console.error("Error saving levels:", error);
      res.status(500).json({ success: false, error: "Failed to save levels" });
    }
  });

  // Save idea name for free user
  app.post("/api/free-user/:sessionId/idea", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const { ideaName } = req.body;
      
      if (!ideaName) {
        return res.status(400).json({ success: false, error: "ideaName is required" });
      }
      
      const existingIdeas = await storage.getUserIdeas(sessionId);
      
      if (existingIdeas.length > 0) {
        const updatedIdea = await storage.updateUserIdea(existingIdeas[0].id, { name: ideaName });
        return res.json({ success: true, idea: updatedIdea });
      } else {
        const newIdea = await storage.createUserIdea({
          sessionId,
          name: ideaName,
        });
        return res.json({ success: true, idea: newIdea });
      }
    } catch (error) {
      console.error("Error saving idea name:", error);
      res.status(500).json({ success: false, error: "Failed to save idea name" });
    }
  });

  // Save display name for free user
  app.post("/api/free-user/:sessionId/display-name", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const { displayName } = req.body;
      
      if (!displayName) {
        return res.status(400).json({ success: false, error: "displayName is required" });
      }
      
      const user = await storage.updateFreeUser(sessionId, { displayName });
      
      if (!user) {
        const newUser = await storage.createFreeUser({
          sessionId,
          displayName,
        });
        return res.json({ success: true, user: newUser, created: true });
      }
      
      res.json({ success: true, user });
    } catch (error) {
      console.error("Error saving display name:", error);
      res.status(500).json({ success: false, error: "Failed to save display name" });
    }
  });

  // Generate i2u.ai registration link for free user
  app.post("/api/free-user/:sessionId/registration-link", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const user = await storage.getFreeUser(sessionId);
      
      if (!user) {
        return res.status(404).json({ success: false, error: "Free user not found" });
      }
      
      // Generate pre-filled registration link
      const params = new URLSearchParams();
      if (user.email) params.set('email', user.email);
      if (user.displayName) params.set('name', user.displayName);
      if (user.primaryStakeholder) params.set('role', user.primaryStakeholder);
      params.set('gleams', user.totalGleams.toString());
      params.set('ref', sessionId);
      
      const registrationUrl = `https://i2u.ai/Register?${params.toString()}`;
      
      // Update registration status
      await storage.updateFreeUser(sessionId, { registrationStatus: 'pending' });
      
      res.json({ success: true, registrationUrl, user });
    } catch (error) {
      console.error("Error generating registration link:", error);
      res.status(500).json({ success: false, error: "Failed to generate registration link" });
    }
  });

  // AI Coaching Chatbot endpoint
  app.post("/api/coaching/chat", async (req, res) => {
    try {
      const { message, context, history } = req.body;

      if (!message) {
        return res.status(400).json({ error: "Message is required" });
      }

      const systemPrompt = `You are an expert AI Unicorn Coach for i2u.ai (Ideas to Unicorns through AI). Your role is to provide personalized startup coaching based on the user's Value Journey Quest assessment.

CONTEXT:
${context ? `- Stakeholder Type: ${context.stakeholder || 'Not specified'}
- Current Level: ${context.currentLevel || 'L0 (Spark)'}
- Aspirational Level: ${context.aspirationalLevel || 'Not set'}
- Total Gleams Earned: ${context.totalGleams || 0}
- Current Score: ${context.currentScore || 'Not assessed'}
- Aspirational Score: ${context.aspirationalScore || 'Not assessed'}` : 'No assessment context available yet.'}

SCORING SYSTEM:
- Gleams (Ğ): Points earned at L0 (Spark level), 10 max per parameter
- Alicorns (🦄): Achievement tokens for L1+ levels, 100 Gleams = 1 Alicorn

9 GROWTH LEVELS (L0-L8):
L0: Spark (Conception) - Resilience focus
L1: Hunt (Initiation) - Validation focus  
L2: Build (Formulation) - Feasibility focus
L3: Launch (Market Entry) - Traction focus
L4: Grow (Scaling) - Growth focus
L5: Profit (Efficiency) - Profit focus
L6: Lead (Leadership) - Innovation focus
L7: Icon (Unicorn) - Legacy focus
L8: Sustain (Stewardship) - Stewardship focus

9 DIMENSIONS per level: D1-D9 covering core competencies
9 EiR (Elephants in the Room): Risk factors that could derail progress

COACHING STYLE:
- Be encouraging but honest about gaps
- Provide actionable insights
- Reference unicorn case studies when relevant
- Help users understand their journey from current to aspirational level
- Celebrate Gleams and Alicorns earned
- Keep responses concise (2-3 paragraphs max)
- Use emojis sparingly for engagement (🦄, ✨, 🚀, 🎯)

FANCY NAME SUGGESTIONS:
When users ask for a fancy name, display name ideas, or cool nickname, suggest creative cosmic/startup themed names like:
- Cosmic Pioneer, Stellar Voyager, Quantum Dreamer, Nova Architect
- Astral Founder, Nebula Builder, Eclipse Visionary, Horizon Seeker
- Starlight Innovator, Galactic Trailblazer, Supernova Creator
- Celestial Pathfinder, Aurora Strategist, Meteor Maverick
Generate 3-5 unique suggestions and explain they can set their fancy name in Settings.

Remember: You're democratizing unicorn building - helping founders from anywhere achieve legendary status!`;

      const chatHistory = (history || []).map((m: any) => ({
        role: m.role as "user" | "assistant",
        content: m.content
      }));

      res.setHeader("Content-Type", "text/event-stream");
      res.setHeader("Cache-Control", "no-cache");
      res.setHeader("Connection", "keep-alive");

      const stream = await openai.chat.completions.create({
        model: "gpt-4o",
        messages: [
          { role: "system", content: systemPrompt },
          ...chatHistory,
          { role: "user", content: message }
        ],
        stream: true,
        max_completion_tokens: 1024,
      });

      for await (const chunk of stream) {
        const content = chunk.choices[0]?.delta?.content || "";
        if (content) {
          res.write(`data: ${JSON.stringify({ content })}\n\n`);
        }
      }

      res.write(`data: ${JSON.stringify({ done: true })}\n\n`);
      res.end();
    } catch (error) {
      console.error("Coaching chat error:", error);
      if (res.headersSent) {
        res.write(`data: ${JSON.stringify({ error: "Failed to get response" })}\n\n`);
        res.end();
      } else {
        res.status(500).json({ error: "Failed to process coaching request" });
      }
    }
  });

  // User name preferences endpoints
  app.get("/api/user/preferences", async (req, res) => {
    try {
      const user = (req as any).user;
      if (!user) {
        return res.json({
          realName: "Unicorn Builder",
          fancyName: "",
          leaderboardNameType: "real",
          shareNameType: "real",
          profileNameType: "real"
        });
      }
      
      const [userData] = await db
        .select()
        .from(users)
        .where(eq(users.id, user.id))
        .limit(1);
      
      const realName = userData?.firstName && userData?.lastName 
        ? `${userData.firstName} ${userData.lastName}`
        : userData?.firstName || "Unicorn Builder";
      
      res.json({
        realName,
        fancyName: userData?.fancyName || "",
        leaderboardNameType: userData?.leaderboardNameType || "real",
        shareNameType: userData?.shareNameType || "real",
        profileNameType: userData?.profileNameType || "real"
      });
    } catch (error) {
      console.error("Error fetching preferences:", error);
      res.status(500).json({ error: "Failed to fetch preferences" });
    }
  });

  app.put("/api/user/preferences", async (req, res) => {
    try {
      const user = (req as any).user;
      if (!user) {
        return res.status(401).json({ error: "Not authenticated" });
      }
      
      const { fancyName, leaderboardNameType, shareNameType, profileNameType } = req.body;
      
      await db
        .update(users)
        .set({
          fancyName: fancyName || null,
          leaderboardNameType: leaderboardNameType || "real",
          shareNameType: shareNameType || "real",
          profileNameType: profileNameType || "real",
          updatedAt: new Date()
        })
        .where(eq(users.id, user.id));
      
      res.json({ success: true });
    } catch (error) {
      console.error("Error updating preferences:", error);
      res.status(500).json({ error: "Failed to update preferences" });
    }
  });

  // Generate fancy name suggestions
  app.get("/api/user/fancy-names", async (req, res) => {
    const suggestions = [
      "Cosmic Pioneer", "Stellar Voyager", "Quantum Dreamer", "Nova Architect",
      "Astral Founder", "Nebula Builder", "Eclipse Visionary", "Horizon Seeker",
      "Starlight Innovator", "Galactic Trailblazer", "Supernova Creator", "Orbit Maker",
      "Celestial Pathfinder", "Aurora Strategist", "Meteor Maverick", "Comet Chaser",
      "Zenith Explorer", "Infinity Weaver", "Prism Catalyst", "Phoenix Ascender"
    ];
    res.json({ suggestions });
  });

  // User Suggestions endpoint
  app.post("/api/suggestions", async (req, res) => {
    try {
      const { suggestion, category = "general" } = req.body;
      
      if (!suggestion || typeof suggestion !== "string" || suggestion.trim().length < 5) {
        return res.status(400).json({ error: "Suggestion must be at least 5 characters" });
      }
      
      const ipAddress = req.headers["x-forwarded-for"] as string || req.socket.remoteAddress || "unknown";
      const sessionId = req.headers["x-session-id"] as string || `session_${Date.now()}`;
      
      const [newSuggestion] = await db
        .insert(userSuggestions)
        .values({
          suggestion: suggestion.trim(),
          category,
          sessionId,
          ipAddress,
          gleamsAwarded: 50,
          status: "pending"
        })
        .returning();
      
      res.json({ 
        success: true, 
        gleamsAwarded: 50,
        suggestionId: newSuggestion.id 
      });
    } catch (error) {
      console.error("Error saving suggestion:", error);
      res.status(500).json({ error: "Failed to save suggestion" });
    }
  });

  // ============================================
  // Multi-Idea & Multi-Role Assessment Routes
  // ============================================

  // Get user's ideas
  app.get("/api/ideas/:sessionId", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const ideas = await storage.getUserIdeas(sessionId);
      res.json({ success: true, ideas });
    } catch (error) {
      console.error("Error fetching ideas:", error);
      res.status(500).json({ error: "Failed to fetch ideas" });
    }
  });

  // Create a new idea
  app.post("/api/ideas", async (req, res) => {
    try {
      const { sessionId, name, description, sector, stage } = req.body;
      
      if (!sessionId || !name) {
        return res.status(400).json({ error: "sessionId and name are required" });
      }
      
      const idea = await storage.createUserIdea({
        sessionId,
        name,
        description,
        sector,
        stage,
      });
      
      res.status(201).json({ success: true, idea });
    } catch (error) {
      console.error("Error creating idea:", error);
      res.status(500).json({ error: "Failed to create idea" });
    }
  });

  // Update an idea
  app.put("/api/ideas/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ error: "Invalid idea ID" });
      }
      
      const updates = req.body;
      const idea = await storage.updateUserIdea(id, updates);
      
      if (!idea) {
        return res.status(404).json({ error: "Idea not found" });
      }
      
      res.json({ success: true, idea });
    } catch (error) {
      console.error("Error updating idea:", error);
      res.status(500).json({ error: "Failed to update idea" });
    }
  });

  // Get roles for an idea
  app.get("/api/ideas/:ideaId/roles", async (req, res) => {
    try {
      const ideaId = parseInt(req.params.ideaId);
      if (isNaN(ideaId)) {
        return res.status(400).json({ error: "Invalid idea ID" });
      }
      
      const roles = await storage.getIdeaRoles(ideaId);
      res.json({ success: true, roles });
    } catch (error) {
      console.error("Error fetching roles:", error);
      res.status(500).json({ error: "Failed to fetch roles" });
    }
  });

  // Get all roles for a user (across all ideas)
  app.get("/api/roles/:sessionId", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const roles = await storage.getAllUserRoles(sessionId);
      res.json({ success: true, roles });
    } catch (error) {
      console.error("Error fetching all roles:", error);
      res.status(500).json({ error: "Failed to fetch roles" });
    }
  });

  // Add a role to an idea
  app.post("/api/roles", async (req, res) => {
    try {
      const { ideaId, sessionId, stakeholder, isPrimary } = req.body;
      
      if (!ideaId || !sessionId || !stakeholder) {
        return res.status(400).json({ error: "ideaId, sessionId, and stakeholder are required" });
      }
      
      const role = await storage.createIdeaRole({
        ideaId,
        sessionId,
        stakeholder,
        isPrimary: isPrimary || false,
      });
      
      res.status(201).json({ success: true, role });
    } catch (error) {
      console.error("Error creating role:", error);
      res.status(500).json({ error: "Failed to create role" });
    }
  });

  // Get assessment sessions for a user
  app.get("/api/assessment-sessions/:sessionId", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const status = req.query.status as string;
      
      let sessions;
      if (status === "paused") {
        sessions = await storage.getPausedAssessmentSessions(sessionId);
      } else if (status === "active") {
        sessions = await storage.getActiveAssessmentSessions(sessionId);
      } else {
        const active = await storage.getActiveAssessmentSessions(sessionId);
        const paused = await storage.getPausedAssessmentSessions(sessionId);
        sessions = [...active, ...paused];
      }
      
      res.json({ success: true, sessions });
    } catch (error) {
      console.error("Error fetching assessment sessions:", error);
      res.status(500).json({ error: "Failed to fetch sessions" });
    }
  });

  // Create or update assessment session
  app.post("/api/assessment-sessions", async (req, res) => {
    try {
      const sessionData = req.body;
      
      if (!sessionData.sessionId || !sessionData.stakeholder || !sessionData.level) {
        return res.status(400).json({ error: "sessionId, stakeholder, and level are required" });
      }
      
      const session = await storage.createAssessmentSession({
        ...sessionData,
        status: sessionData.status || "in_progress",
        assessmentType: sessionData.assessmentType || "current",
      });
      
      res.status(201).json({ success: true, session });
    } catch (error) {
      console.error("Error creating assessment session:", error);
      res.status(500).json({ error: "Failed to create session" });
    }
  });

  // Update assessment session (pause, resume, complete)
  app.put("/api/assessment-sessions/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ error: "Invalid session ID" });
      }
      
      const updates = req.body;
      
      if (updates.status === "paused" && !updates.pausedAt) {
        updates.pausedAt = new Date();
      }
      
      if (updates.status === "completed" && !updates.completedAt) {
        updates.completedAt = new Date();
      }
      
      const session = await storage.updateAssessmentSession(id, updates);
      
      if (!session) {
        return res.status(404).json({ error: "Session not found" });
      }
      
      res.json({ success: true, session });
    } catch (error) {
      console.error("Error updating assessment session:", error);
      res.status(500).json({ error: "Failed to update session" });
    }
  });

  // ===== USER SESSION & MAGIC LINK ENDPOINTS =====

  // Create or get user session with microsecond timestamp
  app.post("/api/user-sessions", async (req, res) => {
    try {
      const { sessionId, userAgent } = req.body;
      
      if (!sessionId) {
        return res.status(400).json({ error: "sessionId is required" });
      }
      
      // Get IP address
      const ipAddress = req.headers['x-forwarded-for']?.toString().split(',')[0].trim() 
        || req.socket.remoteAddress 
        || 'unknown';
      
      // Check if session already exists
      const existing = await db.select().from(userSessions).where(eq(userSessions.sessionId, sessionId)).limit(1);
      
      if (existing.length > 0) {
        // Update last accessed
        await db.update(userSessions)
          .set({ lastAccessedAt: new Date() })
          .where(eq(userSessions.sessionId, sessionId));
        
        res.json({ 
          success: true, 
          session: existing[0],
          isNew: false
        });
        return;
      }
      
      // Create new session with microsecond precision
      const now = new Date();
      const microPart = process.hrtime.bigint().toString().slice(-6);
      const createdAtMicro = now.toISOString().replace('Z', microPart + 'Z');
      
      const [newSession] = await db.insert(userSessions).values({
        sessionId,
        ipAddress,
        createdAtMicro,
        userAgent: userAgent || null,
        isActive: true,
      }).returning();
      
      res.status(201).json({ 
        success: true, 
        session: newSession,
        isNew: true
      });
    } catch (error) {
      console.error("Error creating user session:", error);
      res.status(500).json({ error: "Failed to create session" });
    }
  });

  // Fuzzy name matching utility function
  function fuzzyNameMatch(stored: string, input: string): boolean {
    if (!stored || !input) return false;
    
    // Normalize both strings - lowercase, remove extra spaces
    const normalize = (s: string) => s.toLowerCase().trim().replace(/\s+/g, ' ');
    const storedNorm = normalize(stored);
    const inputNorm = normalize(input);
    
    // Exact match
    if (storedNorm === inputNorm) return true;
    
    // Split into words
    const storedWords = storedNorm.split(' ').filter(w => w.length > 0);
    const inputWords = inputNorm.split(' ').filter(w => w.length > 0);
    
    // Check if any input word matches any stored word (order-independent)
    for (const inputWord of inputWords) {
      for (const storedWord of storedWords) {
        // Exact word match
        if (inputWord === storedWord) return true;
        
        // Fuzzy match with Levenshtein distance for words >= 4 chars
        if (inputWord.length >= 4 && storedWord.length >= 4) {
          const distance = levenshteinDistance(inputWord, storedWord);
          const maxLen = Math.max(inputWord.length, storedWord.length);
          // Allow up to 25% error rate
          if (distance <= Math.ceil(maxLen * 0.25)) return true;
        }
      }
    }
    
    return false;
  }

  // Levenshtein distance for fuzzy matching
  function levenshteinDistance(a: string, b: string): number {
    const matrix: number[][] = [];
    
    for (let i = 0; i <= b.length; i++) {
      matrix[i] = [i];
    }
    for (let j = 0; j <= a.length; j++) {
      matrix[0][j] = j;
    }
    
    for (let i = 1; i <= b.length; i++) {
      for (let j = 1; j <= a.length; j++) {
        if (b.charAt(i - 1) === a.charAt(j - 1)) {
          matrix[i][j] = matrix[i - 1][j - 1];
        } else {
          matrix[i][j] = Math.min(
            matrix[i - 1][j - 1] + 1, // substitution
            matrix[i][j - 1] + 1,     // insertion
            matrix[i - 1][j] + 1      // deletion
          );
        }
      }
    }
    
    return matrix[b.length][a.length];
  }

  // Get user session by sessionId (magic link - returns session info, may require name validation)
  app.get("/api/user-sessions/:sessionId", async (req, res) => {
    try {
      const { sessionId } = req.params;
      
      const sessions = await db.select().from(userSessions)
        .where(and(
          eq(userSessions.sessionId, sessionId),
          eq(userSessions.isActive, true)
        ))
        .limit(1);
      
      if (sessions.length === 0) {
        return res.status(404).json({ error: "Session not found or inactive" });
      }
      
      const session = sessions[0];
      
      // Update last accessed
      await db.update(userSessions)
        .set({ lastAccessedAt: new Date() })
        .where(eq(userSessions.sessionId, sessionId));
      
      // Return session info including whether name validation is required
      res.json({ 
        success: true, 
        session,
        requiresNameValidation: !!session.displayName,
        hasDisplayName: !!session.displayName
      });
    } catch (error) {
      console.error("Error fetching user session:", error);
      res.status(500).json({ error: "Failed to fetch session" });
    }
  });

  // Validate magic link with name (fuzzy matching)
  app.post("/api/user-sessions/:sessionId/validate", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const { name } = req.body;
      
      if (!name || typeof name !== 'string') {
        return res.status(400).json({ error: "Name is required for validation" });
      }
      
      const sessions = await db.select().from(userSessions)
        .where(and(
          eq(userSessions.sessionId, sessionId),
          eq(userSessions.isActive, true)
        ))
        .limit(1);
      
      if (sessions.length === 0) {
        return res.status(404).json({ error: "Session not found or inactive" });
      }
      
      const session = sessions[0];
      
      // If no display name set, this is their first time - set the name
      if (!session.displayName) {
        await db.update(userSessions)
          .set({ displayName: name.trim(), lastAccessedAt: new Date() })
          .where(eq(userSessions.sessionId, sessionId));
        
        return res.json({ 
          success: true, 
          session: { ...session, displayName: name.trim() },
          message: "Name set successfully"
        });
      }
      
      // Validate against stored name with fuzzy matching
      if (!fuzzyNameMatch(session.displayName, name)) {
        return res.status(401).json({ 
          error: "Name doesn't match. Try any word from your name, spelling doesn't have to be perfect!",
          hint: "Enter any word from your fancy name or real name"
        });
      }
      
      // Update last accessed
      await db.update(userSessions)
        .set({ lastAccessedAt: new Date() })
        .where(eq(userSessions.sessionId, sessionId));
      
      res.json({ success: true, session, message: "Validated successfully" });
    } catch (error) {
      console.error("Error validating magic link:", error);
      res.status(500).json({ error: "Failed to validate" });
    }
  });

  // Update display name for a session
  app.put("/api/user-sessions/:sessionId/name", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const { displayName } = req.body;
      
      if (!displayName || typeof displayName !== 'string' || displayName.trim().length === 0) {
        return res.status(400).json({ error: "Display name is required" });
      }
      
      const sessions = await db.select().from(userSessions)
        .where(and(
          eq(userSessions.sessionId, sessionId),
          eq(userSessions.isActive, true)
        ))
        .limit(1);
      
      if (sessions.length === 0) {
        return res.status(404).json({ error: "Session not found" });
      }
      
      await db.update(userSessions)
        .set({ displayName: displayName.trim(), lastAccessedAt: new Date() })
        .where(eq(userSessions.sessionId, sessionId));
      
      res.json({ success: true, message: "Display name updated" });
    } catch (error) {
      console.error("Error updating display name:", error);
      res.status(500).json({ error: "Failed to update display name" });
    }
  });

  // Login with Name credential (device-independent, fuzzy matching)
  // Searches both userSessions and freeUsers tables for matching display names
  app.post("/api/user-sessions/login", async (req, res) => {
    try {
      const { name } = req.body;
      
      if (!name || typeof name !== 'string' || !name.trim()) {
        return res.status(400).json({ error: "Name is required" });
      }
      
      // First, search in freeUsers table (where most display names are stored)
      const freeUsersWithNames = await db.select().from(freeUsers)
        .where(sql`${freeUsers.displayName} IS NOT NULL`);
      
      for (const freeUser of freeUsersWithNames) {
        if (freeUser.displayName && fuzzyNameMatch(freeUser.displayName, name.trim())) {
          // Found a match in freeUsers - create or get a userSession for this
          // First check if there's an existing userSession we can return
          const existingSessions = await db.select().from(userSessions)
            .where(eq(userSessions.isActive, true))
            .limit(1);
          
          if (existingSessions.length > 0) {
            // Update the session with the display name and return it
            await db.update(userSessions)
              .set({ 
                displayName: freeUser.displayName,
                lastAccessedAt: new Date() 
              })
              .where(eq(userSessions.id, existingSessions[0].id));
            
            return res.json({ 
              success: true, 
              session: { ...existingSessions[0], displayName: freeUser.displayName },
              freeUser: freeUser
            });
          }
          
          // If no userSession exists, return the freeUser info with its sessionId
          return res.json({
            success: true,
            session: {
              sessionId: freeUser.sessionId,
              displayName: freeUser.displayName,
              ipAddress: freeUser.ipAddress,
              createdAtMicro: freeUser.createdAt?.toISOString() || new Date().toISOString(),
              isActive: true
            },
            freeUser: freeUser
          });
        }
      }
      
      // Also search in userSessions table
      const sessions = await db.select().from(userSessions)
        .where(and(
          eq(userSessions.isActive, true),
          sql`${userSessions.displayName} IS NOT NULL`
        ));
      
      for (const session of sessions) {
        if (session.displayName && fuzzyNameMatch(session.displayName, name.trim())) {
          await db.update(userSessions)
            .set({ lastAccessedAt: new Date() })
            .where(eq(userSessions.id, session.id));
          
          return res.json({ success: true, session });
        }
      }
      
      return res.status(404).json({ 
        error: "No matching account found. Try any word from your fancy name or real name.",
        hint: "Spelling doesn't have to be perfect!"
      });
    } catch (error) {
      console.error("Error logging in with credentials:", error);
      res.status(500).json({ error: "Failed to verify credentials" });
    }
  });

  // Delete all user data (complete account deletion)
  app.delete("/api/user-data/:sessionId", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const { confirmDelete } = req.body;
      
      if (confirmDelete !== 'DELETE_ALL_MY_DATA') {
        return res.status(400).json({ error: "Please confirm deletion by sending confirmDelete: 'DELETE_ALL_MY_DATA'" });
      }
      
      // Delete in order to respect foreign key constraints
      // 1. Delete assessment sessions
      await db.delete(assessmentSessions).where(eq(assessmentSessions.sessionId, sessionId));
      
      // 2. Get user ideas and delete related data
      const ideas = await db.select().from(userIdeas).where(eq(userIdeas.sessionId, sessionId));
      
      for (const idea of ideas) {
        // Delete idea roles
        await db.delete(ideaRoles).where(eq(ideaRoles.ideaId, idea.id));
      }
      
      // 3. Delete user ideas
      await db.delete(userIdeas).where(eq(userIdeas.sessionId, sessionId));
      
      // 4. Delete free user data
      await db.delete(freeUsers).where(eq(freeUsers.sessionId, sessionId));
      
      // 5. Delete value journey assessments
      await db.delete(valueJourneyAssessments).where(eq(valueJourneyAssessments.sessionId, sessionId));
      
      // 6. Delete user suggestions
      await db.delete(userSuggestions).where(eq(userSuggestions.sessionId, sessionId));
      
      // 7. Finally, mark user session as inactive (keep for audit purposes)
      await db.update(userSessions)
        .set({ isActive: false })
        .where(eq(userSessions.sessionId, sessionId));
      
      res.json({ 
        success: true, 
        message: "All your data has been deleted. Your session has been deactivated." 
      });
    } catch (error) {
      console.error("Error deleting user data:", error);
      res.status(500).json({ error: "Failed to delete user data" });
    }
  });

  // Export all user data
  app.get("/api/user-data/:sessionId/export", async (req, res) => {
    try {
      const { sessionId } = req.params;
      
      // Gather all user data
      const [sessionData] = await db.select().from(userSessions)
        .where(eq(userSessions.sessionId, sessionId)).limit(1);
      
      const [freeUserData] = await db.select().from(freeUsers)
        .where(eq(freeUsers.sessionId, sessionId)).limit(1);
      
      const ideasData = await db.select().from(userIdeas)
        .where(eq(userIdeas.sessionId, sessionId));
      
      const rolesData = await db.select().from(ideaRoles)
        .where(eq(ideaRoles.sessionId, sessionId));
      
      const assessmentSessionsData = await db.select().from(assessmentSessions)
        .where(eq(assessmentSessions.sessionId, sessionId));
      
      const valueJourneyData = await db.select().from(valueJourneyAssessments)
        .where(eq(valueJourneyAssessments.sessionId, sessionId));
      
      const suggestionsData = await db.select().from(userSuggestions)
        .where(eq(userSuggestions.sessionId, sessionId));
      
      const exportData = {
        exportedAt: new Date().toISOString(),
        session: sessionData || null,
        profile: freeUserData || null,
        ideas: ideasData,
        roles: rolesData,
        assessmentSessions: assessmentSessionsData,
        valueJourneyAssessments: valueJourneyData,
        suggestions: suggestionsData,
      };
      
      res.json({ success: true, data: exportData });
    } catch (error) {
      console.error("Error exporting user data:", error);
      res.status(500).json({ error: "Failed to export user data" });
    }
  });

  // ============================================
  // ECOSYSTEM ROUTES
  // ============================================

  // Get ecosystem profiles with optional filters
  app.get("/api/ecosystem/profiles", async (req, res) => {
    try {
      const { entityType, level, sector, country, limit } = req.query;
      const profiles = await storage.getEcosystemProfiles({
        entityType: entityType as string,
        level: level as string,
        sector: sector as string,
        country: country as string,
        limit: limit ? parseInt(limit as string) : undefined,
      });
      res.json({ success: true, profiles, lastUpdated: new Date().toISOString() });
    } catch (error) {
      console.error("Error fetching ecosystem profiles:", error);
      res.status(500).json({ error: "Failed to fetch ecosystem profiles" });
    }
  });

  // Get profile by ID
  app.get("/api/ecosystem/profiles/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ error: "Invalid profile ID" });
      }
      const profile = await storage.getEcosystemProfileById(id);
      if (!profile) {
        return res.status(404).json({ error: "Profile not found" });
      }
      
      // Get team members if it's a startup
      let teamMembers: any[] = [];
      if (profile.entityType === "startup") {
        teamMembers = await storage.getStartupTeamMembers(id);
      }
      
      res.json({ success: true, profile, teamMembers });
    } catch (error) {
      console.error("Error fetching profile:", error);
      res.status(500).json({ error: "Failed to fetch profile" });
    }
  });

  // Create/register ecosystem profile
  app.post("/api/ecosystem/profiles", async (req, res) => {
    try {
      const profileData = req.body;
      if (!profileData.entityType || !profileData.name) {
        return res.status(400).json({ error: "entityType and name are required" });
      }
      
      const profile = await storage.createEcosystemProfile(profileData);
      
      // Create activity for the new profile
      await storage.createEcosystemActivity({
        profileId: profile.id,
        sessionId: profileData.sessionId,
        activityType: "joined",
        title: `${profile.displayName || profile.name} joined the ecosystem`,
        description: `A new ${profile.entityType.replace('_', ' ')} has joined at ${profile.level}`,
      });
      
      res.status(201).json({ success: true, profile });
    } catch (error) {
      console.error("Error creating profile:", error);
      res.status(500).json({ error: "Failed to create profile" });
    }
  });

  // Update ecosystem profile
  app.put("/api/ecosystem/profiles/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ error: "Invalid profile ID" });
      }
      
      const profile = await storage.updateEcosystemProfile(id, req.body);
      if (!profile) {
        return res.status(404).json({ error: "Profile not found" });
      }
      
      res.json({ success: true, profile });
    } catch (error) {
      console.error("Error updating profile:", error);
      res.status(500).json({ error: "Failed to update profile" });
    }
  });

  // Get leaderboard by level
  app.get("/api/ecosystem/leaderboard/level/:level", async (req, res) => {
    try {
      const { level } = req.params;
      const limit = req.query.limit ? parseInt(req.query.limit as string) : 20;
      const profiles = await storage.getLeaderboardByLevel(level, limit);
      res.json({ success: true, profiles, level, lastUpdated: new Date().toISOString() });
    } catch (error) {
      console.error("Error fetching leaderboard:", error);
      res.status(500).json({ error: "Failed to fetch leaderboard" });
    }
  });

  // Get leaderboard by entity type
  app.get("/api/ecosystem/leaderboard/type/:entityType", async (req, res) => {
    try {
      const { entityType } = req.params;
      const limit = req.query.limit ? parseInt(req.query.limit as string) : 20;
      const profiles = await storage.getLeaderboardByEntityType(entityType, limit);
      res.json({ success: true, profiles, entityType, lastUpdated: new Date().toISOString() });
    } catch (error) {
      console.error("Error fetching leaderboard:", error);
      res.status(500).json({ error: "Failed to fetch leaderboard" });
    }
  });

  // Get recent activity feed
  app.get("/api/ecosystem/activity", async (req, res) => {
    try {
      const limit = req.query.limit ? parseInt(req.query.limit as string) : 50;
      const activities = await storage.getRecentActivity(limit);
      res.json({ success: true, activities, lastUpdated: new Date().toISOString() });
    } catch (error) {
      console.error("Error fetching activity:", error);
      res.status(500).json({ error: "Failed to fetch activity" });
    }
  });

  // Create activity
  app.post("/api/ecosystem/activity", async (req, res) => {
    try {
      const activity = await storage.createEcosystemActivity(req.body);
      res.status(201).json({ success: true, activity });
    } catch (error) {
      console.error("Error creating activity:", error);
      res.status(500).json({ error: "Failed to create activity" });
    }
  });

  // Find matches for a profile
  app.get("/api/ecosystem/matches/:profileId", async (req, res) => {
    try {
      const profileId = parseInt(req.params.profileId);
      if (isNaN(profileId)) {
        return res.status(400).json({ error: "Invalid profile ID" });
      }
      
      const entityTypes = req.query.entityTypes ? (req.query.entityTypes as string).split(',') : undefined;
      const levels = req.query.levels ? (req.query.levels as string).split(',') : undefined;
      const sectors = req.query.sectors ? (req.query.sectors as string).split(',') : undefined;
      
      const matches = await storage.findMatches(profileId, { entityTypes, levels, sectors });
      res.json({ success: true, matches, lastUpdated: new Date().toISOString() });
    } catch (error) {
      console.error("Error finding matches:", error);
      res.status(500).json({ error: "Failed to find matches" });
    }
  });

  // Create match connection
  app.post("/api/ecosystem/connections", async (req, res) => {
    try {
      const { fromProfileId, toProfileId, initiatedBy, matchReason } = req.body;
      if (!fromProfileId || !toProfileId) {
        return res.status(400).json({ error: "fromProfileId and toProfileId are required" });
      }
      
      const connection = await storage.createMatchConnection({
        fromProfileId,
        toProfileId,
        initiatedBy,
        matchReason,
      });
      
      res.status(201).json({ success: true, connection });
    } catch (error) {
      console.error("Error creating connection:", error);
      res.status(500).json({ error: "Failed to create connection" });
    }
  });

  // Get connections for a profile
  app.get("/api/ecosystem/connections/:profileId", async (req, res) => {
    try {
      const profileId = parseInt(req.params.profileId);
      if (isNaN(profileId)) {
        return res.status(400).json({ error: "Invalid profile ID" });
      }
      
      const connections = await storage.getMatchConnections(profileId);
      res.json({ success: true, connections });
    } catch (error) {
      console.error("Error fetching connections:", error);
      res.status(500).json({ error: "Failed to fetch connections" });
    }
  });

  // Get cached news
  app.get("/api/ecosystem/news", async (req, res) => {
    try {
      const limit = req.query.limit ? parseInt(req.query.limit as string) : 20;
      const news = await storage.getCachedNews(limit);
      res.json({ success: true, news, lastUpdated: new Date().toISOString() });
    } catch (error) {
      console.error("Error fetching news:", error);
      res.status(500).json({ error: "Failed to fetch news" });
    }
  });

  // Add team member to startup
  app.post("/api/ecosystem/profiles/:profileId/team", async (req, res) => {
    try {
      const profileId = parseInt(req.params.profileId);
      if (isNaN(profileId)) {
        return res.status(400).json({ error: "Invalid profile ID" });
      }
      
      const { name, role, linkedinUrl, imageUrl, bio } = req.body;
      if (!name || !role) {
        return res.status(400).json({ error: "name and role are required" });
      }
      
      const member = await storage.createStartupTeamMember({
        profileId,
        name,
        role,
        linkedinUrl,
        imageUrl,
        bio,
      });
      
      res.status(201).json({ success: true, member });
    } catch (error) {
      console.error("Error adding team member:", error);
      res.status(500).json({ error: "Failed to add team member" });
    }
  });

  // Get ecosystem dashboard summary
  app.get("/api/ecosystem/dashboard", async (req, res) => {
    try {
      // Get counts and top performers
      const [startups, investors, mentors, corporates, accelerators] = await Promise.all([
        storage.getEcosystemProfiles({ entityType: "startup", limit: 10 }),
        storage.getEcosystemProfiles({ entityType: "investor_vc", limit: 10 }),
        storage.getEcosystemProfiles({ entityType: "mentor", limit: 10 }),
        storage.getEcosystemProfiles({ entityType: "corporate", limit: 10 }),
        storage.getEcosystemProfiles({ entityType: "accelerator", limit: 10 }),
      ]);
      
      const activities = await storage.getRecentActivity(20);
      const news = await storage.getCachedNews(10);
      
      res.json({
        success: true,
        dashboard: {
          startups,
          investors,
          mentors,
          corporates,
          accelerators,
          activities,
          news,
        },
        lastUpdated: new Date().toISOString(),
        nextRefresh: new Date(Date.now() + 60 * 60 * 1000).toISOString(), // 1 hour from now
      });
    } catch (error) {
      console.error("Error fetching dashboard:", error);
      res.status(500).json({ error: "Failed to fetch dashboard" });
    }
  });

  return httpServer;
}
