import express, { type Express } from "express";
import fs from "fs";
import path from "path";

export function serveStatic(app: Express) {
  const distPath = path.resolve(__dirname, "public");
  if (!fs.existsSync(distPath)) {
    throw new Error(
      `Could not find the build directory: ${distPath}, make sure to build the client first`,
    );
  }

  const indexHtmlPath = path.resolve(distPath, "index.html");

  // Serve static assets with long cache (they have hashed filenames)
  // But exclude index.html - we'll serve that dynamically
  app.use(express.static(distPath, {
    maxAge: '1y',
    immutable: true,
    index: false,
    setHeaders: (res, filePath) => {
      if (filePath.endsWith('.html')) {
        res.setHeader('Cache-Control', 'private, no-store, max-age=0, must-revalidate');
        res.setHeader('Surrogate-Control', 'no-store');
        res.setHeader('Pragma', 'no-cache');
        res.setHeader('Expires', '0');
      }
    }
  }));

  // Serve index.html dynamically - read fresh on EVERY request with unique timestamp
  app.use("*", (_req, res) => {
    // Read fresh HTML on each request
    let indexHtml = fs.readFileSync(indexHtmlPath, 'utf-8');
    
    // Inject a per-request timestamp to make content unique
    const requestTimestamp = Date.now();
    indexHtml = indexHtml.replace(
      '</head>',
      `<meta name="request-id" content="${requestTimestamp}" />\n</head>`
    );
    
    // Aggressive anti-cache headers
    res.setHeader('Cache-Control', 'private, no-store, no-cache, max-age=0, must-revalidate');
    res.setHeader('Surrogate-Control', 'no-store');
    res.setHeader('Pragma', 'no-cache');
    res.setHeader('Expires', '0');
    res.setHeader('Vary', '*');
    res.setHeader('Content-Type', 'text/html; charset=utf-8');
    res.send(indexHtml);
  });
}
