import { 
  type Assessment, type InsertAssessment, 
  type LeaderboardEntry, type InsertLeaderboardEntry, 
  type UserAssessmentProgress, type InsertUserAssessmentProgress,
  type LevelCompletion, type InsertLevelCompletion,
  type FreeUser, type InsertFreeUser,
  type UserIdea, type InsertUserIdea,
  type IdeaRole, type InsertIdeaRole,
  type AssessmentSession, type InsertAssessmentSession,
  type PauseState,
  type EcosystemProfile, type InsertEcosystemProfile,
  type EcosystemActivity, type InsertEcosystemActivity,
  type MatchingPreferences, type InsertMatchingPreferences,
  type MatchConnection, type InsertMatchConnection,
  type CachedNews, type InsertCachedNews,
  type StartupTeamMember, type InsertStartupTeamMember,
  assessments, leaderboardEntries, userAssessmentProgress, levelCompletions, freeUsers,
  userIdeas, ideaRoles, assessmentSessions, ecosystemProfiles, ecosystemActivity,
  matchingPreferences, matchConnections, cachedNews, startupTeamMembers
} from "@shared/schema";
import { drizzle } from "drizzle-orm/postgres-js";
import postgres from "postgres";
import { eq, desc, and, or, sql, ilike, inArray } from "drizzle-orm";

export interface IStorage {
  createAssessment(assessment: InsertAssessment): Promise<Assessment>;
  getAssessmentById(id: number): Promise<Assessment | undefined>;
  createLeaderboardEntry(entry: InsertLeaderboardEntry): Promise<LeaderboardEntry>;
  getLeaderboardEntries(limit?: number, phase?: number): Promise<LeaderboardEntry[]>;
  
  // User Assessment Progress
  getUserProgress(userId: string): Promise<UserAssessmentProgress | undefined>;
  createUserProgress(progress: InsertUserAssessmentProgress): Promise<UserAssessmentProgress>;
  updateUserProgress(userId: string, updates: Partial<InsertUserAssessmentProgress>): Promise<UserAssessmentProgress | undefined>;
  updatePauseState(userId: string, pauseState: PauseState | null): Promise<void>;
  requestAdditionalStakeholder(userId: string, stakeholder: string): Promise<void>;
  
  // Level Completions
  getLevelCompletions(userId: string): Promise<LevelCompletion[]>;
  createLevelCompletion(completion: InsertLevelCompletion): Promise<LevelCompletion>;
  
  // Free Users
  getFreeUser(sessionId: string): Promise<FreeUser | undefined>;
  createFreeUser(user: InsertFreeUser): Promise<FreeUser>;
  updateFreeUser(sessionId: string, updates: Partial<InsertFreeUser>): Promise<FreeUser | undefined>;
  getFreeUserByEmail(email: string): Promise<FreeUser | undefined>;
  
  // User Ideas
  getUserIdeas(sessionId: string): Promise<UserIdea[]>;
  createUserIdea(idea: InsertUserIdea): Promise<UserIdea>;
  updateUserIdea(id: number, updates: Partial<InsertUserIdea>): Promise<UserIdea | undefined>;
  
  // Idea Roles
  getIdeaRoles(ideaId: number): Promise<IdeaRole[]>;
  getAllUserRoles(sessionId: string): Promise<IdeaRole[]>;
  createIdeaRole(role: InsertIdeaRole): Promise<IdeaRole>;
  
  // Assessment Sessions
  getAssessmentSession(sessionId: string, ideaId: number | null, level: string): Promise<AssessmentSession | undefined>;
  getActiveAssessmentSessions(sessionId: string): Promise<AssessmentSession[]>;
  getPausedAssessmentSessions(sessionId: string): Promise<AssessmentSession[]>;
  createAssessmentSession(session: InsertAssessmentSession): Promise<AssessmentSession>;
  updateAssessmentSession(id: number, updates: Partial<InsertAssessmentSession>): Promise<AssessmentSession | undefined>;
  
  // Ecosystem Profiles
  getEcosystemProfiles(filters?: { entityType?: string; level?: string; sector?: string; country?: string; limit?: number }): Promise<EcosystemProfile[]>;
  getEcosystemProfileById(id: number): Promise<EcosystemProfile | undefined>;
  getEcosystemProfileBySessionId(sessionId: string): Promise<EcosystemProfile | undefined>;
  createEcosystemProfile(profile: InsertEcosystemProfile): Promise<EcosystemProfile>;
  updateEcosystemProfile(id: number, updates: Partial<InsertEcosystemProfile>): Promise<EcosystemProfile | undefined>;
  getLeaderboardByLevel(level: string, limit?: number): Promise<EcosystemProfile[]>;
  getLeaderboardByEntityType(entityType: string, limit?: number): Promise<EcosystemProfile[]>;
  
  // Ecosystem Activity
  getRecentActivity(limit?: number): Promise<EcosystemActivity[]>;
  createEcosystemActivity(activity: InsertEcosystemActivity): Promise<EcosystemActivity>;
  
  // Matching
  getMatchingPreferences(profileId: number): Promise<MatchingPreferences | undefined>;
  createMatchingPreferences(prefs: InsertMatchingPreferences): Promise<MatchingPreferences>;
  updateMatchingPreferences(profileId: number, updates: Partial<InsertMatchingPreferences>): Promise<MatchingPreferences | undefined>;
  findMatches(profileId: number, filters?: { entityTypes?: string[]; levels?: string[]; sectors?: string[] }): Promise<EcosystemProfile[]>;
  createMatchConnection(connection: InsertMatchConnection): Promise<MatchConnection>;
  getMatchConnections(profileId: number): Promise<MatchConnection[]>;
  
  // News
  getCachedNews(limit?: number): Promise<CachedNews[]>;
  createCachedNews(news: InsertCachedNews): Promise<CachedNews>;
  clearOldNews(): Promise<void>;
  
  // Team Members
  getStartupTeamMembers(profileId: number): Promise<StartupTeamMember[]>;
  createStartupTeamMember(member: InsertStartupTeamMember): Promise<StartupTeamMember>;
}

const connectionString = process.env.DATABASE_URL!;
const client = postgres(connectionString);
export const db = drizzle(client);

export class DatabaseStorage implements IStorage {
  async createAssessment(insertAssessment: InsertAssessment): Promise<Assessment> {
    const result = await db.insert(assessments).values(insertAssessment).returning();
    return result[0];
  }

  async getAssessmentById(id: number): Promise<Assessment | undefined> {
    const result = await db.select().from(assessments).where(eq(assessments.id, id)).limit(1);
    return result[0];
  }

  async createLeaderboardEntry(entry: InsertLeaderboardEntry): Promise<LeaderboardEntry> {
    const result = await db.insert(leaderboardEntries).values(entry).returning();
    return result[0];
  }

  async getLeaderboardEntries(limit: number = 50, phase?: number): Promise<LeaderboardEntry[]> {
    if (phase) {
      return await db.select()
        .from(leaderboardEntries)
        .where(eq(leaderboardEntries.phase, phase))
        .orderBy(desc(leaderboardEntries.score))
        .limit(limit);
    }
    return await db.select()
      .from(leaderboardEntries)
      .orderBy(desc(leaderboardEntries.score))
      .limit(limit);
  }

  // User Assessment Progress methods
  async getUserProgress(userId: string): Promise<UserAssessmentProgress | undefined> {
    const result = await db.select()
      .from(userAssessmentProgress)
      .where(eq(userAssessmentProgress.userId, userId))
      .limit(1);
    return result[0];
  }

  async createUserProgress(progress: InsertUserAssessmentProgress): Promise<UserAssessmentProgress> {
    const result = await db.insert(userAssessmentProgress).values(progress).returning();
    return result[0];
  }

  async updateUserProgress(userId: string, updates: Partial<InsertUserAssessmentProgress>): Promise<UserAssessmentProgress | undefined> {
    const result = await db.update(userAssessmentProgress)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(userAssessmentProgress.userId, userId))
      .returning();
    return result[0];
  }

  async updatePauseState(userId: string, pauseState: PauseState | null): Promise<void> {
    await db.update(userAssessmentProgress)
      .set({ currentPauseState: pauseState, updatedAt: new Date() })
      .where(eq(userAssessmentProgress.userId, userId));
  }

  async requestAdditionalStakeholder(userId: string, stakeholder: string): Promise<void> {
    const progress = await this.getUserProgress(userId);
    if (progress) {
      const currentRequests = progress.additionalStakeholderRequests || [];
      if (!currentRequests.includes(stakeholder)) {
        await db.update(userAssessmentProgress)
          .set({ 
            additionalStakeholderRequests: [...currentRequests, stakeholder],
            updatedAt: new Date() 
          })
          .where(eq(userAssessmentProgress.userId, userId));
      }
    }
  }

  // Level Completions methods
  async getLevelCompletions(userId: string): Promise<LevelCompletion[]> {
    return await db.select()
      .from(levelCompletions)
      .where(eq(levelCompletions.userId, userId))
      .orderBy(desc(levelCompletions.completedAt));
  }

  async createLevelCompletion(completion: InsertLevelCompletion): Promise<LevelCompletion> {
    const result = await db.insert(levelCompletions).values(completion).returning();
    return result[0];
  }

  // Free Users methods
  async getFreeUser(sessionId: string): Promise<FreeUser | undefined> {
    const result = await db.select()
      .from(freeUsers)
      .where(eq(freeUsers.sessionId, sessionId))
      .limit(1);
    return result[0];
  }

  async createFreeUser(user: InsertFreeUser): Promise<FreeUser> {
    const result = await db.insert(freeUsers).values(user).returning();
    return result[0];
  }

  async updateFreeUser(sessionId: string, updates: Partial<InsertFreeUser>): Promise<FreeUser | undefined> {
    const result = await db.update(freeUsers)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(freeUsers.sessionId, sessionId))
      .returning();
    return result[0];
  }

  async getFreeUserByEmail(email: string): Promise<FreeUser | undefined> {
    const result = await db.select()
      .from(freeUsers)
      .where(eq(freeUsers.email, email))
      .limit(1);
    return result[0];
  }

  // User Ideas methods
  async getUserIdeas(sessionId: string): Promise<UserIdea[]> {
    return await db.select()
      .from(userIdeas)
      .where(and(eq(userIdeas.sessionId, sessionId), eq(userIdeas.isActive, true)))
      .orderBy(desc(userIdeas.updatedAt));
  }

  async createUserIdea(idea: InsertUserIdea): Promise<UserIdea> {
    const result = await db.insert(userIdeas).values(idea).returning();
    return result[0];
  }

  async updateUserIdea(id: number, updates: Partial<InsertUserIdea>): Promise<UserIdea | undefined> {
    const result = await db.update(userIdeas)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(userIdeas.id, id))
      .returning();
    return result[0];
  }

  // Idea Roles methods
  async getIdeaRoles(ideaId: number): Promise<IdeaRole[]> {
    return await db.select()
      .from(ideaRoles)
      .where(and(eq(ideaRoles.ideaId, ideaId), eq(ideaRoles.isActive, true)));
  }

  async getAllUserRoles(sessionId: string): Promise<IdeaRole[]> {
    return await db.select()
      .from(ideaRoles)
      .where(and(eq(ideaRoles.sessionId, sessionId), eq(ideaRoles.isActive, true)));
  }

  async createIdeaRole(role: InsertIdeaRole): Promise<IdeaRole> {
    const result = await db.insert(ideaRoles).values(role).returning();
    return result[0];
  }

  // Assessment Sessions methods
  async getAssessmentSession(sessionId: string, ideaId: number | null, level: string): Promise<AssessmentSession | undefined> {
    if (ideaId) {
      const result = await db.select()
        .from(assessmentSessions)
        .where(and(
          eq(assessmentSessions.sessionId, sessionId),
          eq(assessmentSessions.ideaId, ideaId),
          eq(assessmentSessions.level, level)
        ))
        .limit(1);
      return result[0];
    } else {
      const result = await db.select()
        .from(assessmentSessions)
        .where(and(
          eq(assessmentSessions.sessionId, sessionId),
          eq(assessmentSessions.level, level)
        ))
        .limit(1);
      return result[0];
    }
  }

  async getActiveAssessmentSessions(sessionId: string): Promise<AssessmentSession[]> {
    return await db.select()
      .from(assessmentSessions)
      .where(and(
        eq(assessmentSessions.sessionId, sessionId),
        eq(assessmentSessions.status, "in_progress")
      ))
      .orderBy(desc(assessmentSessions.lastActivityAt));
  }

  async getPausedAssessmentSessions(sessionId: string): Promise<AssessmentSession[]> {
    return await db.select()
      .from(assessmentSessions)
      .where(and(
        eq(assessmentSessions.sessionId, sessionId),
        eq(assessmentSessions.status, "paused")
      ))
      .orderBy(desc(assessmentSessions.pausedAt));
  }

  async createAssessmentSession(session: InsertAssessmentSession): Promise<AssessmentSession> {
    const result = await db.insert(assessmentSessions).values(session).returning();
    return result[0];
  }

  async updateAssessmentSession(id: number, updates: Partial<InsertAssessmentSession>): Promise<AssessmentSession | undefined> {
    const result = await db.update(assessmentSessions)
      .set({ ...updates, lastActivityAt: new Date() })
      .where(eq(assessmentSessions.id, id))
      .returning();
    return result[0];
  }

  // ========== ECOSYSTEM METHODS ==========

  async getEcosystemProfiles(filters?: { entityType?: string; level?: string; sector?: string; country?: string; limit?: number }): Promise<EcosystemProfile[]> {
    let query = db.select().from(ecosystemProfiles).where(eq(ecosystemProfiles.isPublic, true));
    
    const conditions: any[] = [eq(ecosystemProfiles.isPublic, true)];
    
    if (filters?.entityType) {
      conditions.push(eq(ecosystemProfiles.entityType, filters.entityType));
    }
    if (filters?.level) {
      conditions.push(eq(ecosystemProfiles.level, filters.level));
    }
    if (filters?.sector) {
      conditions.push(eq(ecosystemProfiles.sector, filters.sector));
    }
    if (filters?.country) {
      conditions.push(eq(ecosystemProfiles.country, filters.country));
    }
    
    return await db.select()
      .from(ecosystemProfiles)
      .where(and(...conditions))
      .orderBy(desc(ecosystemProfiles.totalAlicorns))
      .limit(filters?.limit || 50);
  }

  async getEcosystemProfileById(id: number): Promise<EcosystemProfile | undefined> {
    const result = await db.select()
      .from(ecosystemProfiles)
      .where(eq(ecosystemProfiles.id, id))
      .limit(1);
    return result[0];
  }

  async getEcosystemProfileBySessionId(sessionId: string): Promise<EcosystemProfile | undefined> {
    const result = await db.select()
      .from(ecosystemProfiles)
      .where(eq(ecosystemProfiles.sessionId, sessionId))
      .limit(1);
    return result[0];
  }

  async createEcosystemProfile(profile: InsertEcosystemProfile): Promise<EcosystemProfile> {
    const result = await db.insert(ecosystemProfiles).values(profile).returning();
    return result[0];
  }

  async updateEcosystemProfile(id: number, updates: Partial<InsertEcosystemProfile>): Promise<EcosystemProfile | undefined> {
    const result = await db.update(ecosystemProfiles)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(ecosystemProfiles.id, id))
      .returning();
    return result[0];
  }

  async getLeaderboardByLevel(level: string, limit: number = 20): Promise<EcosystemProfile[]> {
    return await db.select()
      .from(ecosystemProfiles)
      .where(and(
        eq(ecosystemProfiles.level, level),
        eq(ecosystemProfiles.isPublic, true)
      ))
      .orderBy(desc(ecosystemProfiles.totalAlicorns))
      .limit(limit);
  }

  async getLeaderboardByEntityType(entityType: string, limit: number = 20): Promise<EcosystemProfile[]> {
    return await db.select()
      .from(ecosystemProfiles)
      .where(and(
        eq(ecosystemProfiles.entityType, entityType),
        eq(ecosystemProfiles.isPublic, true)
      ))
      .orderBy(desc(ecosystemProfiles.totalAlicorns))
      .limit(limit);
  }

  // Activity Feed
  async getRecentActivity(limit: number = 50): Promise<EcosystemActivity[]> {
    return await db.select()
      .from(ecosystemActivity)
      .where(eq(ecosystemActivity.isPublic, true))
      .orderBy(desc(ecosystemActivity.createdAt))
      .limit(limit);
  }

  async createEcosystemActivity(activity: InsertEcosystemActivity): Promise<EcosystemActivity> {
    const result = await db.insert(ecosystemActivity).values(activity).returning();
    return result[0];
  }

  // Matching
  async getMatchingPreferences(profileId: number): Promise<MatchingPreferences | undefined> {
    const result = await db.select()
      .from(matchingPreferences)
      .where(eq(matchingPreferences.profileId, profileId))
      .limit(1);
    return result[0];
  }

  async createMatchingPreferences(prefs: InsertMatchingPreferences): Promise<MatchingPreferences> {
    const result = await db.insert(matchingPreferences).values(prefs).returning();
    return result[0];
  }

  async updateMatchingPreferences(profileId: number, updates: Partial<InsertMatchingPreferences>): Promise<MatchingPreferences | undefined> {
    const result = await db.update(matchingPreferences)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(matchingPreferences.profileId, profileId))
      .returning();
    return result[0];
  }

  async findMatches(profileId: number, filters?: { entityTypes?: string[]; levels?: string[]; sectors?: string[] }): Promise<EcosystemProfile[]> {
    const conditions: any[] = [
      eq(ecosystemProfiles.isPublic, true),
      sql`${ecosystemProfiles.id} != ${profileId}`
    ];
    
    if (filters?.entityTypes && filters.entityTypes.length > 0) {
      conditions.push(inArray(ecosystemProfiles.entityType, filters.entityTypes));
    }
    if (filters?.levels && filters.levels.length > 0) {
      conditions.push(inArray(ecosystemProfiles.level, filters.levels));
    }
    if (filters?.sectors && filters.sectors.length > 0) {
      conditions.push(inArray(ecosystemProfiles.sector, filters.sectors));
    }
    
    return await db.select()
      .from(ecosystemProfiles)
      .where(and(...conditions))
      .orderBy(desc(ecosystemProfiles.totalAlicorns))
      .limit(50);
  }

  async createMatchConnection(connection: InsertMatchConnection): Promise<MatchConnection> {
    const result = await db.insert(matchConnections).values(connection).returning();
    return result[0];
  }

  async getMatchConnections(profileId: number): Promise<MatchConnection[]> {
    return await db.select()
      .from(matchConnections)
      .where(or(
        eq(matchConnections.fromProfileId, profileId),
        eq(matchConnections.toProfileId, profileId)
      ))
      .orderBy(desc(matchConnections.createdAt));
  }

  // News
  async getCachedNews(limit: number = 20): Promise<CachedNews[]> {
    return await db.select()
      .from(cachedNews)
      .where(eq(cachedNews.isActive, true))
      .orderBy(desc(cachedNews.fetchedAt))
      .limit(limit);
  }

  async createCachedNews(news: InsertCachedNews): Promise<CachedNews> {
    const result = await db.insert(cachedNews).values(news).returning();
    return result[0];
  }

  async clearOldNews(): Promise<void> {
    const oneDayAgo = new Date(Date.now() - 24 * 60 * 60 * 1000);
    await db.delete(cachedNews)
      .where(sql`${cachedNews.fetchedAt} < ${oneDayAgo}`);
  }

  // Team Members
  async getStartupTeamMembers(profileId: number): Promise<StartupTeamMember[]> {
    return await db.select()
      .from(startupTeamMembers)
      .where(eq(startupTeamMembers.profileId, profileId));
  }

  async createStartupTeamMember(member: InsertStartupTeamMember): Promise<StartupTeamMember> {
    const result = await db.insert(startupTeamMembers).values(member).returning();
    return result[0];
  }
}

export const storage = new DatabaseStorage();
